# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2021 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENCE.TXT
#
import os

from clcommon.const import Feature
from clcommon.utils import ExternalProgramFailed, get_passenger_package_name, is_litespeed_running
from clconfig.ui_config_lib import get_ui_config
from clwizard.constants import CL_SELECTOR_BIN, MODULES_LOGS_DIR
from clwizard.exceptions import InstallationFailedException, PackageMissingError
from clwizard.utils import installed_interpreters_list

from .base import WizardInstaller


class PythonInstaller(WizardInstaller):
    LOG_FILE = os.path.join(MODULES_LOGS_DIR, 'python.log')
    _REQUIRED_CL_COMPONENT_SUPPORT = Feature.PYTHON_SELECTOR

    def _install_python_versions(self, versions):
        """
        Install given python versions
        :param versions: versions of python packages
        """
        groups = []
        for v in versions:
            package_name = 'alt-python' + v.replace('.', '')
            groups.append(package_name)
        self.app_logger.info("Try to install Python package with versions: %s", ','.join(versions))
        try:
            self._install_groups(*groups)
        except ExternalProgramFailed as e:
            raise InstallationFailedException() from e

    def _configure_for_litespeed_server(self):
        """
        In order to make python apps work on Litespeed server
        we need to do run extra script
        see docs for details:
        https://www.litespeedtech.com/support/wiki/doku.php/litespeed_wiki:cloudlinux:enable_passenger
        :return:
        """
        litespeed_script = '/usr/local/lsws/admin/misc/enable_ruby_python_selector.sh'
        # we are going to simplify process of making python/ruby apps work under Litespeed
        # so in future we will not need to run extra script
        if not os.path.isfile(litespeed_script):
            return
        try:
            self._run_command(['sh', litespeed_script])
        except ExternalProgramFailed as e:
            raise InstallationFailedException() from e

    @staticmethod
    def _get_python_default_version():
        # type: () -> str
        try:
            from clselect.clselectctl import get_default_version  # pylint: disable=import-outside-toplevel
        except ImportError as e:
            raise PackageMissingError('lvemanager') from e
        return get_default_version('python')

    def _set_python_default_version(self, version):
        # type: (str) -> None
        self.app_logger.info("trying to set default python version as '%s'", version)
        try:
            self._run_command(
                [
                    CL_SELECTOR_BIN,
                    'set',
                    '--interpreter',
                    'python',
                    '--default-version',
                    version,
                    '--json',
                ]
            )
        except ExternalProgramFailed as e:
            raise InstallationFailedException() from e

    def _is_already_configured(self):
        """
        Check that python selector is ready to work
        All configurations were done:
        - passenger package is installed
        We always have python27 installed,
        so we do not need to check if at least one interpreter is installed
        If the config file does not exist, we ignore the uiSettings
        :return: already_configured status
        :rtype: bool
        """
        return self._is_package_installed(get_passenger_package_name()) and (
            not get_ui_config() or not get_ui_config().get('uiSettings', {}).get('hidePythonApp')
        )

    def initial_status(self):
        installed_python_versions = []
        interpreters_list = installed_interpreters_list('python')
        for interpreter in interpreters_list:
            if interpreter.installed:
                installed_python_versions.append(interpreter.version)
        return {
            'already_configured': self._is_already_configured(),
            'options': {
                'installed_versions': installed_python_versions,
                'available_versions': self._get_available_versions('python'),
                'default_version': self._get_python_default_version(),
            },
        }

    def run_installation(self, options):
        install_versions = options.get('versions', [])
        default_version = options.get('default_version')

        if default_version and default_version not in install_versions:
            self.app_logger.error(
                "Version %s that specified to be set as default "
                "must be included in install_versions",
                default_version,
            )
            raise InstallationFailedException()

        self._install_passenger()
        self._install_python_versions(install_versions)
        # if Litespeed server is running - do some extra configuration
        if is_litespeed_running():
            self._configure_for_litespeed_server()
        # do not hide python app in web interface
        try:
            self._run_command(
                [
                    'cloudlinux-config',
                    'set',
                    '--json',
                    '--data={"options": {"uiSettings": {"hidePythonApp": false}}}',
                ]
            )
        except ExternalProgramFailed as e:
            raise InstallationFailedException() from e

        if default_version:
            self._set_python_default_version(default_version)

    @classmethod
    def supported_options(cls):
        return {'versions', 'default_version'}
