# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2021 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENCE.TXT
#
import os
from typing import Dict, List  # NOQA

from clcommon.const import Feature
from clcommon.utils import ExternalProgramFailed
from clconfig.cagefs_statistics_config import check_cagefs_initialized
from clwizard.constants import CL_SELECTOR_BIN, MODULES_LOGS_DIR
from clwizard.exceptions import InstallationFailedException, PackageMissingError

from .base import WizardInstaller


class PhpInstaller(WizardInstaller):
    LOG_FILE = os.path.join(MODULES_LOGS_DIR, 'php.log')
    _REQUIRED_CL_COMPONENT_SUPPORT = Feature.PHP_SELECTOR

    def _set_default_php_version(self, version):
        # type: (str) -> None
        self.app_logger.info("trying to set default php version as '%s'", version)
        try:
            self._run_command(
                [
                    CL_SELECTOR_BIN,
                    'set',
                    '--interpreter',
                    'php',
                    '--default-version',
                    version,
                    '--json',
                ]
            )
        except ExternalProgramFailed as e:
            raise InstallationFailedException() from e

    def _set_cloudlinux_selector_status(self, status):
        # type: (str) -> None
        self.app_logger.info("trying to set PHP Selector state '%s'", status)
        try:
            self._run_command(
                [
                    CL_SELECTOR_BIN,
                    'set',
                    '--interpreter',
                    'php',
                    '--selector-status',
                    status,
                    '--json',
                ]
            )
        except ExternalProgramFailed as e:
            raise InstallationFailedException() from e

    def _install_php_version(self, versions):
        # type: (List) -> None
        """
        Install given php versions via groups
        :param versions: e.g ['4.4', '5.4', '5.6']
        :return: None
        """
        self.app_logger.info(
            "php version '%s' is going to be installed", ','.join(versions)
        )
        groups = []
        for ver in versions:
            # given version in format '5.6' get it's package name
            package_name = 'alt-php' + ver.replace('.', '')
            groups.append(package_name)
        try:
            # with RPM we install php versions via groups,
            # all extra alt-phpXX modules will be installed as dependencies

            # On Ubuntu, _install_groups will install the meta packages.
            self._install_groups(*groups)
        except ExternalProgramFailed as e:
            raise InstallationFailedException() from e

    def run_installation(self, options):
        #  type: (Dict) -> None
        """
        Run installation of php module
        It is possible to install only alt-php packages and skip further
        installation if 'enable_selector': False
        Otherwise we will configure php selector also
        -- set selector status to enabled
        -- set default version if it was passed
        :param options: {'versions': [5.5, 5.6], 'enable_selector': True}
        :return:
        """
        default_version = options.get('default_version')
        install_versions = options.get('versions', [])
        to_enable_selector = options.get('enable_selector')

        if to_enable_selector:
            if default_version and default_version not in install_versions:
                self.app_logger.error(
                    "Version %s that was specified to be set as default "
                    "must be included in install_versions",
                    default_version,
                )
                raise InstallationFailedException()

        self._install_php_version(install_versions)

        if check_cagefs_initialized():
            if to_enable_selector:
                self._set_cloudlinux_selector_status(status='enabled')

                if default_version:
                    self._set_default_php_version(default_version)
            else:
                try:
                    self._set_cloudlinux_selector_status(status='disabled')
                except Exception as e:
                    self.app_logger.error(
                        "Error occurred while disabling PHP Selector: %s", str(e)
                    )

    @classmethod
    def supported_options(cls):
        return {'default_version', 'versions', 'enable_selector'}

    @staticmethod
    def _php_get_default_version():
        # type: () -> str
        try:
            from clselect.clselect import ClSelect  # pylint: disable=import-outside-toplevel
        except ImportError as e:
            raise PackageMissingError('lvemanager') from e
        return ClSelect().get_version()[0]

    @staticmethod
    def _php_interpreters_list():
        # type: () -> List
        try:
            from clselect.clselectctl import (  # pylint: disable=import-outside-toplevel
                interpreter_versions_short_summary
            )
        except ImportError as e:
            raise PackageMissingError('lvemanager') from e
        return interpreter_versions_short_summary('php')

    def _get_warnings(self):
        # type: () -> List[Dict]
        """
        Get list of warnings that should be shown in wizard
        before module installation
        """
        warnings = [
            {
                # this message is hardcoded because we do not want to
                # show a lot of possible warnings, so just tell the client to go
                # to the settings page where cldiag warnings are shown
                'message': "Further CloudLinux PHP Selector setup will be available "
                "in LVE Manager settings after installation. Find more details "
                "in our documentation at %(url)s.",
                'context': {
                    'url': 'https://docs.cloudlinux.com/php_selector_installation.html'
                },
            }
        ]

        return warnings

    def initial_status(self):
        # type: () -> Dict

        installed_versions = []
        for php_interpreter in self._php_interpreters_list():
            if not php_interpreter.installed:
                continue
            installed_versions.append(php_interpreter.version)
        return {
            # As 'PHP Configure' in the wizard could be used for our PHP selector AND
            # for installing PHP interpreters, 'already_configured' should be True
            # when any PHP interpreter is installed.
            'already_configured': any(installed_versions),
            'options': {
                'default_version': self._php_get_default_version(),
                'installed_versions': installed_versions,
                'available_versions': self._get_available_versions('php'),
                'enable_selector': self._is_php_selector_enabled(),
            },
            'warnings': self._get_warnings(),
        }

    def _is_php_selector_enabled(self):
        # type: () -> bool
        """
        Return True if Php Selector enabled in UI and could be used
        """
        return bool(check_cagefs_initialized() and self._is_php_selector_enabled_in_config())

    def _is_php_selector_enabled_in_config(self):
        # type: () -> bool
        """
        Return True if Php Selector is enabled for UI in panel specific (DA, Plesk or cPanel) config
        """
        self.app_logger.info("trying to get Php Selector state.")
        try:
            from clselector.selectorlib import CloudlinuxSelectorLib  # pylint: disable=import-outside-toplevel

            return CloudlinuxSelectorLib('php').php_selector_is_enabled()
        except ImportError:
            return False
