# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2021 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENCE.TXT
#
import os
from typing import Dict, Iterable  # NOQA

from clcommon.const import Feature
from clcommon.utils import ExternalProgramFailed, get_passenger_package_name
from clwizard.constants import CL_SELECTOR_BIN, MODULES_LOGS_DIR
from clwizard.exceptions import InstallationFailedException, PackageMissingError
from clwizard.modules.base import WizardInstaller
from clwizard.utils import installed_interpreters_list


class NodejsInstaller(WizardInstaller):
    LOG_FILE = os.path.join(MODULES_LOGS_DIR, 'nodejs.log')
    _REQUIRED_CL_COMPONENT_SUPPORT = Feature.NODEJS_SELECTOR

    def _set_default_nodejs_version(self, version):
        # type: (str) -> None
        self.app_logger.info("trying to set default NodeJS version as '%s'", version)
        try:
            self._run_command(
                [
                    CL_SELECTOR_BIN,
                    'set',
                    '--interpreter',
                    'nodejs',
                    '--default-version',
                    version,
                    '--json',
                ]
            )
        except ExternalProgramFailed as e:
            raise InstallationFailedException() from e

    def _set_cloudlinux_selector_status(self, status):
        # type: (str) -> None
        self.app_logger.info("trying to set NodeJS Selector state '%s'", status)
        try:
            self._run_command(
                [
                    CL_SELECTOR_BIN,
                    'set',
                    '--interpreter',
                    'nodejs',
                    '--selector-status',
                    status,
                    '--json',
                ]
            )
        except ExternalProgramFailed as e:
            raise InstallationFailedException() from e

    def _install_nodejs_versions(self, versions):
        # type: (Iterable[str]) -> None
        self.app_logger.info("Trying to install NodeJS versions: %s", ', '.join(versions))
        # Given a version (e.g., '6') construct required package group name
        group_names = ['alt-nodejs' + version for version in versions]
        try:
            self._install_groups(*group_names)
        except ExternalProgramFailed as e:
            raise InstallationFailedException() from e

    def run_installation(self, options):
        default_version = options.get('default_version')  # type: str
        install_versions = options.get('versions', [])

        if default_version and default_version not in install_versions:
            self.app_logger.error(
                "Version %s that was specified to be set as default "
                "must be included in install_versions",
                default_version,
            )
            raise InstallationFailedException()

        self._install_passenger()
        self._install_nodejs_versions(install_versions)
        self._set_cloudlinux_selector_status(status='enabled')

        if default_version:
            self._set_default_nodejs_version(default_version)

    @classmethod
    def supported_options(cls):
        return {'default_version', 'versions'}

    @staticmethod
    def _nodejs_get_default_version():
        # type: () -> str
        try:
            from clselect.clselectctl import get_default_version  # pylint: disable=import-outside-toplevel
        except ImportError as e:
            raise PackageMissingError('lvemanager') from e
        return get_default_version('nodejs')

    def _is_already_configured(self):
        # type: () -> bool
        """
        Checks that nodejs selector is ready to work
        All configurations were done:
        - At least one interpreter version is installed
        - Passenger is installed
        - Nodejs selector is enabled
        :return: bool value configured or not
        """
        return bool(
            any(it.installed for it in installed_interpreters_list('nodejs'))
            and self._is_package_installed(get_passenger_package_name())
            and self._is_nodejs_selector_enabled()
        )

    def initial_status(self):
        # type: () -> Dict
        """
        Return dictionary with two required keys: installed and options.
        This will be used by lvemanager UI to properly display wizard.
        """
        interpreters_list = installed_interpreters_list('nodejs')
        installed_versions = [it.version for it in interpreters_list if it.installed]
        return {
            'already_configured': self._is_already_configured(),
            'options': {
                'default_version': self._nodejs_get_default_version(),
                'installed_versions': installed_versions,
                'available_versions': self._get_available_versions('nodejs'),
            },
        }

    def _is_nodejs_selector_enabled(self):
        # type: () -> bool
        """
        Return True if Nodejs Selector enabled
        :return: bool
        """
        self.app_logger.info("trying to get NodeJS Selector state.")
        try:
            from clselector.selectorlib import CloudlinuxSelectorLib  # pylint: disable=import-outside-toplevel

            return CloudlinuxSelectorLib('nodejs').get_nodejs_selector_status()['selector_enabled']
        except ImportError:
            return False
