# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2021 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENCE.TXT
#
from dataclasses import dataclass
import os
from contextlib import contextmanager
from typing import Any, Dict, List  # NOQA

from clcommon.const import Feature
from clcommon.utils import ExternalProgramFailed, is_ea4, is_litespeed_running, is_testing_enabled_repo, is_ubuntu
from cldetectlib import get_apache_modules, get_boolean_param, is_da
from clwizard.constants import MODULES_LOGS_DIR
from clwizard.exceptions import InstallationFailedException

from .base import WizardInstaller

_DA_CUSTOMBUILD = "/usr/local/directadmin/custombuild/build"


# LSAPI package naming schemes differ between normal CL and Ubuntu.
# This is why we define two sets of package names here.
@dataclass
class LsapiPackageSet:
    general: List[str]  # general LSAPI-related packages
    general_lsapi: str  # LSAPI package for systems without EA4 (not cPanel)
    ea4_lsapi: str  # EA4 lsapi package


cl_lsapi_set = LsapiPackageSet(
    general=["liblsapi", "liblsapi-devel"],
    general_lsapi="mod_lsapi",
    ea4_lsapi="ea-apache24-mod_lsapi",
)

ubuntu_lsapi_set = LsapiPackageSet(
    general=["liblsapi", "liblsapi-dev"],
    general_lsapi="mod-lsapi",
    ea4_lsapi="ea-apache24-mod-lsapi",
)


class LsapiInstaller(WizardInstaller):
    LOG_FILE = os.path.join(MODULES_LOGS_DIR, "lsapi.log")

    UTILITY = "/usr/bin/switch_mod_lsapi"
    _REQUIRED_CL_COMPONENT_SUPPORT = Feature.LSAPI

    def __init__(self):
        super().__init__()
        self._prepare_lsapi_packages()
        self.apache_modules = get_apache_modules() or []  # type: List[str]

    def _prepare_lsapi_packages(self):
        package_set = ubuntu_lsapi_set if is_ubuntu() else cl_lsapi_set
        self.lsapi_packages = package_set.general[:]

        if is_ea4():
            self.lsapi_packages.append(package_set.ea4_lsapi)
        else:
            self.lsapi_packages.append(package_set.general_lsapi)

    def _install_lsapi_packages(self):
        """
        Install lsapi packages if needed
        :return: None
        """
        packages_to_install = []
        for package in self.lsapi_packages:
            if not self._is_package_installed(package):
                packages_to_install.append(package)
        if packages_to_install:
            try:
                out = self._install_package(*packages_to_install)
            except ExternalProgramFailed as err:
                self.app_logger.error("Package installation failed with error: %s", str(err))
                raise InstallationFailedException() from err
            self.app_logger.info("Package was installed successfully: %s", out)
        else:
            self.app_logger.info("Skip package installation, all packages are already installed")

    def _initialize_lsapi(self):
        """
        Configure lsapi on base package installation.
        :return: None
        """
        try:
            self._run_command([self.UTILITY, "--setup"])
        except ExternalProgramFailed as err:
            raise InstallationFailedException() from err

    @contextmanager
    def use_cl_beta_on_da(self, use_beta: bool):
        """
        Context manager to set the option `cloudlinux_beta` in DA custombuild options.
        @param use_beta: True == `cloudlinux_beta=yes`, False == `cloudlinux_beta=no`
        """

        inital_state = get_boolean_param("/usr/local/directadmin/custombuild/options.conf", "cloudlinux_beta")
        if use_beta and not inital_state:
            self._run_command([_DA_CUSTOMBUILD, "set", "cloudlinux_beta", "yes"])
        try:
            yield
        finally:
            if use_beta and not inital_state:
                self._run_command([_DA_CUSTOMBUILD, "set", "cloudlinux_beta", "no"])

    def _install_lsapi_on_da(self, use_beta: bool) -> None:
        """
        Configure and build lsapi on DirectAdmin.

        :param use_beta: Whether to use the beta version of CloudLinux packages.
        :return: None
        """
        try:
            with self.use_cl_beta_on_da(use_beta=use_beta):
                self._run_command([_DA_CUSTOMBUILD, "update"])
                self._run_command([_DA_CUSTOMBUILD, "set", "php1_mode", "lsphp"])
                self._run_command([_DA_CUSTOMBUILD, "php", "n"])
                self._run_command([_DA_CUSTOMBUILD, "apache"])
        except ExternalProgramFailed as err:
            raise InstallationFailedException() from err

    def run_installation(self, options):
        """
        Install the lsapi module on the system.
        :return: None
        """
        if is_da():
            if "use_beta_for_da" in options:
                use_beta_for_da = options["use_beta_for_da"]
            else:
                use_beta_for_da = is_testing_enabled_repo()
            self._install_lsapi_on_da(use_beta_for_da)
        else:
            self._install_lsapi_packages()
            self._initialize_lsapi()

    @classmethod
    def supported_options(cls):
        return {"use_beta_for_da"}

    def _get_warnings(self) -> List[Dict[str, str | dict[str, str]]]:
        """
        Get a list of warnings that should be shown in Wizard before the module installation.

        :return: List of warnings to be shown in UI.
        Warning is a dictionary with the following keys and values:
        - message: str - warning message with placeholders for additional context
        - context: dict - additional context for the message
        """
        warnings: List[Dict[str, str | dict[str, str]]] = []
        if is_da():
            warnings.append({"message": "Installation will be performed via DirectAdmin CustomBuild tool"})
        if "suexec_module" not in self.apache_modules:
            warnings.append(
                {
                    "message": (
                        "mod_suexec is not installed."
                        " It is recommended to use mod_suexec with mod_lsapi."
                        " It is also required for CRIU to work."
                        " Please see %(url)s for more information."
                    ),
                    "context": {"url": "https://docs.cloudlinux.com/apache_mod_lsapi.html"},
                }
            )
        return warnings

    def _get_blockers(self) -> List[Dict[str, str | dict[str, str]]]:
        """
        Get a list of possible blockers that may disable module in Wizard UI.

        :return: List of blockers to be shown in UI.
        Blocker is a dictionary with the following keys and values:
        - message: str - blocker message with placeholders for additional context
        - context: dict - additional context for the message
        """
        blockers = []
        if is_litespeed_running():
            blockers.append(
                {
                    "message": "The server is running under Litespeed."
                    " mod_lsapi works with the Apache server only."
                    " Please see %(url)s for the requirements.",
                    "context": {"url": "https://docs.cloudlinux.com/apache_mod_lsapi.html"},
                }
            )
        if "ruid2_module" in self.apache_modules:
            blockers.append(
                {
                    "message": "mod_ruid2 is enabled, it is not compatible with mod_lsapi."
                    " Please see %(url)s for the requirements.",
                    "context": {
                        "url": "https://docs.cloudlinux.com/apache_mod_lsapi.html",
                    },
                }
            )
        if "mpm_itk_module" in self.apache_modules:
            blockers.append(
                {
                    "message": "MPM ITK is enabled, it is not compatible with mod_lsapi."
                    " Please see %(url)s for the requirements.",
                    "context": {
                        "url": "https://docs.cloudlinux.com/apache_mod_lsapi.html",
                    },
                }
            )
        return blockers

    def initial_status(self):
        result: Dict[str, Any] = {
            "already_configured": all(self._is_package_installed(pkg) for pkg in self.lsapi_packages),
        }

        warnings = self._get_warnings()
        if warnings:
            result.update({"warnings": warnings})
        blockers = self._get_blockers()
        if blockers:
            result.update({"blockers": blockers})
        return result
