#!/opt/cloudlinux/venv/bin/python3 -bb
# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2021 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENCE.TXT
#
"""
Config for wizard is not like all other our configs,
it can be read and modified in multiple processes at the
same time, so we should handle that situations correctly.

Thus, we must set up exclusive locks for reading
and writing and use contextmanager to load config on start and save
right after leaving scope and in case of any errors.

You can use config like this:
with config.acquire_config_access() as config:
   options = config.get_options(module)
or
with config.acquire_config_access() as config:
   options = config.set_state(module, 'failed')
"""
import fcntl
from contextlib import contextmanager

from typing import Optional  # NOQA

from clwizard.config.config import Config
from clwizard.constants import MODULES_STATUS_FILE_LOCK
from clwizard.config.exceptions import UnableAcquireLockError
from .exceptions import (
    BaseConfigError,
    NoSuchModule,
    MalformedConfigError,
)

__all__ = (
    'BaseConfigError',
    'NoSuchModule',
    'MalformedConfigError',
    'acquire_config_access'
)

# We used the contextmanager which provides access to the config object because:
# - making Config class take a lock automatically means less flexibility
#   e.g. sometimes it is required to do complex operations atomically, like in wizard.py:
#   with acquire_config_access() as config:
#       ******
#       if options is not None:
#           config.set_modules(options)
#       ****
#       worker_pid = call_func(...)
#       ****
#       config.worker_pid = worker_pid
#   we should save config only when both, set_modules and worker_pid are done
#   otherwise config will be 'broken'; doing same stuff by automatically locking set_*
#   methods is quite tricky; you may say that we can move this contextmanager inside
#   the Config class, but first see reason #2 below
#
# - direct access to Config class may spawn code that initializes config in class __init__
#   which may lead to the following problems:
#   1. too many config re-reads in case when we do this automatically on each get_* call
#   2. invalid data in cache if we add some flag like 'reread' or something like that

_config = None  # type: Optional[Config]


@contextmanager
def acquire_config_access():
    global _config
    # case when config required in nested context
    if _config is not None:
        yield _config
        return

    try:
        lock_file = open(MODULES_STATUS_FILE_LOCK, 'w', encoding='utf-8')
    except (IOError, OSError) as e:
        raise UnableAcquireLockError(error_message=str(e)) from e

    # wait for exclusive lock
    fcntl.flock(lock_file, fcntl.LOCK_EX)
    try:
        _config = Config()
        yield _config
    finally:
        # to avoid lock leak in case of config errors
        try:
            # Config() could fail.
            _config is None or _config.save()
        finally:
            _config = None
            fcntl.flock(lock_file, fcntl.LOCK_UN)
