# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2021 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENCE.TXT
#
import argparse
import json

from clwizard.modules import get_supported_modules


def validate_json_modules(json_string):
    try:
        data = json.loads(json_string)
    except (TypeError, ValueError) as json_error:
        raise argparse.ArgumentTypeError("Invalid JSON data passed") from json_error

    supported_modules = get_supported_modules()

    # TODO: Validate given json data separately for each module
    for name, instructions in data.items():
        if name not in supported_modules:
            raise argparse.ArgumentTypeError(
                f"'{name}' module is not supported at the moment, "
                f"available keys: {list(supported_modules.keys())}"
            )
        if "options" in list(instructions.keys()):
            if not set(instructions["options"].keys()).issubset(
                    supported_modules[name].supported_options()
            ):
                raise argparse.ArgumentTypeError(
                    f"'{name}' module has incorrect options: {list(instructions['options'].keys())}"
                )
        elif instructions:
            raise argparse.ArgumentTypeError(
                f"'{name}' module has incorrect instructions: {list(instructions.keys())}"
            )
    return data


def parse_cloudlinux_wizard_opts(argv):
    """
    Arg parser for cloudlinux-wizard utility
    """
    parser = argparse.ArgumentParser(
        prog="cloudlinux-wizard",
        description="Utility for installing cloudlinux modules",
    )
    # --json parameter is a convenience for UI calls, we do not process it
    parser.add_argument("--json", action="store_true", help=argparse.SUPPRESS)
    subparser = parser.add_subparsers(dest="subparser")

    install_parser = subparser.add_parser(
        "install",
        help="Install a module. The installation is performed in the background by default",
    )
    install_parser.add_argument(
        "--no-async", help="Install in the foreground", action="store_true"
    )
    install_parser.add_argument(
        "--force",
        help="Force the installer to ignore existing installations",
        action="store_true",
    )

    installer_group = install_parser.add_mutually_exclusive_group(required=True)
    installer_group.add_argument(
        "--json-data",
        help="JSON with modules to be installed",
        type=validate_json_modules,
    )
    installer_group.add_argument(
        "--resume",
        help="Restart the installation process with previously given arguments",
        action="store_true",
    )

    status_parser = subparser.add_parser(
        "status", help="Return JSON with statuses of all modules"
    )
    status_parser.add_argument(
        "--initial", "-i", action="store_true", help="Get the current modules' status"
    )

    finish_parser = subparser.add_parser(
        "finish", help="Finalize the installation from UI and close the wizard"
    )

    cancel_parser = subparser.add_parser(
        "cancel", help="Remove a module from the installation queue"
    )
    cancel_parser.add_argument("module", choices=list(get_supported_modules()))

    opts = parser.parse_args(argv)
    return opts
