#!/usr/bin/python
# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

import os
from xml.dom import minidom as xml
from xml.parsers.expat import ExpatError

from lxml import etree

from .ve_lock import CONFIG_DIR, acquire_lock

# Path to ve.cfg
VE_CFG_PATH = os.path.join(CONFIG_DIR, 've.cfg')
VE_CFG_PATH_TMP = VE_CFG_PATH + '.tmp'

# It's effectively a tuple of ve_cfg and it's lve_config
PARSED_XML_CACHE = (None, None,)


class BadVeConfigException(Exception):
    def __init__(self, message='Nothing'):
        super().__init__(
            f"Unable to parse /etc/container/ve.cfg file, message: {message}"
        )


def get_xml_config(use_cache=True):
    """
    Load xml config from ve.cfg
    1. setup lock on file
    2. read info
    3. remove lock
    :type use_cache: bool
    :rtype: tuple[xml.Document, xml.Element]
    :raises: BadVeConfigException, LockFailedException
    """
    global PARSED_XML_CACHE

    if PARSED_XML_CACHE != (None, None,) and use_cache:
        # Cache will be updated in save_xml()
        return PARSED_XML_CACHE

    with acquire_lock():
        try:
            _ve_cfg = xml.parse(VE_CFG_PATH)
            _ve_lveconfig = _ve_cfg.getElementsByTagName("lveconfig")[0]
        except (OSError, IOError, ExpatError, IndexError) as e:
            raise BadVeConfigException('bad ve.cfg file') from e

        if use_cache:
            PARSED_XML_CACHE = _ve_cfg, _ve_lveconfig
        return _ve_cfg, _ve_lveconfig


def get_xml_config_etree():
    # type: () -> etree.ElementBase
    """
    This method works same as get_xml_config,
    but returns etree object
    :rtype: etree.ElementTree
    :raises: BadVeConfigException, LockFailedException
    """
    with acquire_lock():
        try:
            tree = etree.parse(VE_CFG_PATH)
            return tree.getroot()
        except etree.ParseError as e:
            raise BadVeConfigException(str(e)) from e


def save_xml(xml_document):
    # Update cache on save because we sometimes do re-read right after save
    global PARSED_XML_CACHE
    _ve_lveconfig = xml_document.getElementsByTagName("lveconfig")[0]
    PARSED_XML_CACHE = (xml_document, _ve_lveconfig,)

    xml_string = xml_document.toprettyxml(encoding='utf-8', indent='', newl='')
    xml_string = xml_string.replace(b"\n", b'').replace(b"\t", b'')
    new_xml = xml.parseString(xml_string)
    buf = new_xml.toprettyxml(encoding='utf-8')
    if not buf:     # paranoia mode)
        raise RuntimeError(f"Rejected attempt to write empty {VE_CFG_PATH}")
    with open(VE_CFG_PATH_TMP, 'wb') as f:
        f.write(buf)
        f.flush()
        os.fsync(f.fileno())
    os.rename(VE_CFG_PATH_TMP, VE_CFG_PATH)
