# coding=utf-8

# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT
import logging
import sentry_sdk

from clcommon.utils import get_rhn_systemid_value, get_username
from clcommon.lib.network import get_ip_addr, get_hostname

from clsentry.client import (
    UserlandClient,
    SafeRequestsHTTPTransportSentrySdk,
    ThreadedHttpTransport,
    get_user_tags,
    before_send
)
from clsentry.utils import get_pkg_version

from raven.handlers.logging import SentryHandler
from raven.conf import setup_logging
from sentry_sdk.integrations.logging import LoggingIntegration

CLLIB_DSN = 'https://9713d1296f804031b058b8f2d789d7ac:8ddacae32d8246cf8b25cf826bf3fc0a@cl.sentry.cloudlinux.com/12'
__all__ = ('init_sentry_client', 'init_cllib_sentry_client', 'init_sentry_sdk_client')


def init_sentry_client(project, release, dsn, handle=True, custom_length=None):
    """
    @deprecated use init_sentry_sdk_client
    Create generic sentry client and install logging hooks
    """
    sentry = UserlandClient(dsn, release=release)
    # set user context settings, like id or email
    sentry.user_context({
        'id': get_rhn_systemid_value('system_id') or get_ip_addr(get_hostname()) or get_hostname() or get_username()
    })
    # and also set project name
    sentry.tags['Project'] = project

    if custom_length:
        sentry.string_max_length = custom_length

    if handle:
        # setup handler, so we can track logging.error's
        handler = SentryHandler(sentry, level=logging.ERROR)
        setup_logging(handler)
    return sentry


def init_cllib_sentry_client():
    """
    Create sentry client for cllib package
    and install logging hooks
    """
    return init_sentry_client(
        'python-cllib', release=get_pkg_version('alt-python27-cllib'), dsn=CLLIB_DSN)


def init_sentry_sdk_client(project, release, dsn, handle=True, custom_length=1000,
                           transport=SafeRequestsHTTPTransportSentrySdk, environment=None):
    """
    Initialize the Sentry SDK client and configure logging integration.

    :param project: Project name.
    :param release: Application release version.
    :param dsn: Sentry DSN.
    :param handle: Enable logging integration (default: True).
    :param custom_length: Max length for captured values (default: 1000).
    :param transport: Transport method, can be 'threading' or a transport class.
    :param environment: Environment name (default: None). If None or empty string,
                        defaults to production environment.
    :return: Configured Sentry SDK instance.
    """

    sentry_sdk.init(
        dsn=dsn,
        release=release,
        environment=environment or None,
        transport=ThreadedHttpTransport if transport == 'threading' else transport,
        ignore_errors=[KeyboardInterrupt],
        max_value_length=custom_length,
        before_send=before_send,
        attach_stacktrace=True,
        _experiments={"auto_enabling_integrations": False},
        integrations=[LoggingIntegration(level=logging.ERROR, event_level=logging.ERROR)] if handle else [],
    )

    with sentry_sdk.configure_scope() as scope:
        scope.set_user({
            'id': get_rhn_systemid_value('system_id') or get_ip_addr(get_hostname()) or get_hostname() or get_username()
        })
        for key, value in get_user_tags().items():
            scope.set_tag(key, value)
        scope.set_tag("Project", project)

    return sentry_sdk


def sentry_sdk_send_message(message: str, level: str = "info", tags: dict = None, extra: dict = None,
                            attachments: list[dict] = None, user: dict = None, fingerprint: list = None,
                            contexts: dict = None, transaction: str = None):
    """
    Send a message to Sentry with additional context.

    :param message: The text of the message.
    :param level: Logging level (debug, info, warning, error, fatal).
    :param tags: Additional tags (dict). For example: {"foo": "bar"}
    :param extra: Additional data (dict). For example: {"foo": "bar"}
    :param attachments: Additional files (list). With "path" or "bytes", "filename", and optional "content_type.
    :param user: User information (dict). With "id", and "username".
    :param fingerprint: Fingerprint information (list). For example: ["my-custom-events-group"]
    :param contexts: Additional context data (dict). For example: {"character": {"age": 19}}
    :param transaction: Transaction name (string). For example: UserListView

    Docs: https://docs.sentry.io/platforms/python/enriching-events/
    """

    with sentry_sdk.push_scope() as scope:
        if tags:
            for key, value in tags.items():
                scope.set_tag(key, value)

        if extra:
            for key, value in extra.items():
                scope.set_extra(key, value)

        if attachments:
            for attachment in attachments:
                scope.add_attachment(**attachment)

        if user:
            scope.set_user(user)

        if fingerprint:
            scope.fingerprint = fingerprint

        if contexts:
            for key, value in contexts.items():
                scope.set_context(key, value)

        if transaction:
            scope.set_transaction_name(transaction)

        sentry_sdk.capture_message(message, level=level)
