# -*- coding: utf-8 -*-

# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

from __future__ import print_function
from __future__ import division
from __future__ import absolute_import
import glob
import os
import re
import subprocess

from clcommon import clcagefs
from clselect.clselectexcept import ClSelectExcept
from clselect.utils import run_process_in_cagefs


ALT_PATH = '/opt/alt'
INT_BASE = 'python'
INT_VERS = '??'
DIGITS_VERSION_PATTERN = re.compile(r'(?P<version>\d\.?\d+)$')
VERSION_PATTERN = re.compile(r'(^|.*/)python(?P<version>\d\.\d+)$')

WRAPPERS_PATH = '/usr/share/l.v.e-manager/utils'
PYTHON_WRAPPER = 'python_wrapper'


class Interpreter(object):

    def __init__(self, prefix=None, binary=None, version=None,
                 version_full=None, target_user=None):
        self.binary_list = list()  # accumulate list python bin in virtual environment (using for update python bin)
        self.user = target_user
        if prefix and binary:
            self.prefix = prefix
            self.binary = binary
            self.binary_list.append(self.binary)
            self.python_bin = binary
        elif prefix:
            self.python_bin = None
            self.prefix = prefix
            match = re.search(DIGITS_VERSION_PATTERN, self.prefix)
            suffix = match.group('version')
            if '.' not in suffix:
                suffix = f'{suffix[0]}.{suffix[1:]}'
            self.version_from_suffix = suffix

            python_realbin = os.path.join(self.prefix, 'bin', 'python')
            binaries_to_check = [
                python_realbin,  # bin/python
                python_realbin + suffix.split('.')[0],  # bin/python2
                python_realbin + suffix   # bin/python2.*
            ]
            for bin_file in binaries_to_check:
                if self._is_real_file(bin_file):
                    self.python_bin = bin_file
                    self.binary_list.append(bin_file)
                if self._is_link_to_wrapper(bin_file):
                    binary = bin_file

                real_binary = bin_file + "_bin"
                if self._is_real_file(real_binary):
                    self.python_bin = real_binary
                    self.binary_list.append(real_binary)

            # now python_bin is REAL BINARY in venv
            # for old selector: binary pythonX.Y
            # for new selector: binary pythonX.Y_bin
            if not self.python_bin:
                raise ClSelectExcept.InterpreterError(
                    'Can not find python binary in directory: "%s"' % os.path.dirname(python_realbin))

            # if there is no symlink to wrapper (old selector)
            if not binary:
                binary = self.python_bin
            self.binary = binary

        elif binary:
            self.binary = binary
            self.binary_list = [binary]
            self.prefix = os.path.abspath(os.path.join(
                os.path.dirname(binary), os.path.pardir))
            self.python_bin = binary
        else:
            interpreter = interpreters('version')['2.7']
            self.binary = interpreter.binary
            self.prefix = interpreter.prefix
            self.version = interpreter.version
            self.version_full = interpreter.version_full
            self.python_bin = interpreter.binary

        if version:
            self.version = version
        else:
            self.version = None

        if version_full:
            self.version_full = version_full
        else:
            try:
                if self.user and not clcagefs.in_cagefs():
                    user = self.user
                    result = run_process_in_cagefs(user, self.python_bin,
                                                   ['-c', 'import sys; print(sys.version)'])
                    if result['returncode'] != 0:
                        raise OSError('Unable to get python version %s', result['output'])
                    self.version_full = result['output'].split()[0]
                else:
                    # already in cagefs OR no need to exec as user (e.g global alt python is called)
                    version_raw = subprocess.check_output(
                        [self.python_bin, '-c', 'import sys; print(sys.version)'], text=True)
                    self.version_full = version_raw.split()[0]
            except (OSError, IndexError, subprocess.CalledProcessError) as e:
                ver_ = VERSION_PATTERN.match(self.binary)
                if ver_:
                    self.version = ver_.group('version')
                    self.version_full = None
                elif self.version_from_suffix:
                    self.version = self.version_from_suffix
                    self.version_full = None
                else:
                    raise ClSelectExcept.InterpreterError(
                        'Error detect python binary version "%s"; %s' % (self.binary, str(e)))

        if not self.version and self.version_full:
            self.version = '.'.join(self.version_full.split('.')[:2])

    def _is_real_file(self, path):
        """Check that given path is a file and not symlink"""
        return os.path.isfile(path) and (not os.path.islink(path) or os.readlink(path).startswith('/opt/alt/python'))

    @staticmethod
    def _is_link_to_wrapper(path):
        """
        Since we have python wrapper, all python files are symlinks now
        So real binary - path which must be written to .htaccess
        is considered symlink that links to python wrapper
        """
        wrapper_path = os.path.join(WRAPPERS_PATH, PYTHON_WRAPPER)
        if os.path.islink(path):
            return os.readlink(path) == wrapper_path
        return os.path.isfile(path)

    def __repr__(self):
        return (
            "%s.%s(prefix='%s', version='%s', binary='%s', "
            "version_full='%s')" % (
                self.__class__.__module__, self.__class__.__name__,
                self.prefix, self.version, self.binary, self.version_full
            ))

    def as_dict(self, key=None):
        i = {
            'prefix': self.prefix,
            'binary': self.binary,
            'version': self.version,
            'version_full': self.version_full,
            }
        if key:
            del i[key]
            return {getattr(self, key): i}
        return i


def interpreters(key=None):
    search_path = os.path.join(ALT_PATH, INT_BASE)
    digit = '[0-9]'
    interps = []
    for path in glob.glob(f'{search_path}{digit * 2}') + glob.glob(f'{search_path}{digit * 3}'):
        try:
            # global interpreters -> no need to pass user
            interpreter = Interpreter(path)
        except ClSelectExcept.InterpreterError:
            continue
        # the only case when version_full is None is
        # when we are not able to get it from binary
        # (e.g. we are in broken virtual environment or
        # some package is broken), and we do not need such
        # interpreters neither in statistics nor in selector
        if interpreter.version_full is not None:
            interps.append(interpreter)
    if key:
        return dict((getattr(i, key), i) for i in interps)
    else:
        return interps


def interpreters_dict(key):
    return dict(list(i.as_dict(key).items())[0] for i in interpreters())
