# -*- coding: utf-8 -*-

# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

from __future__ import print_function
from __future__ import division
from __future__ import absolute_import
import os
import re

from clselect.clselectexcept import ClSelectExcept


class Interpreter(object):

    def __init__(self, prefix, versions_info=None):
        """
        :param prefix: mandatory prefix for nodejs root
        :param versions_info: optional precomputed dict with handy info
                              about versions
        """
        if versions_info is None:
            from clselect.clselectnodejs import scan_node_versions
            versions_info = scan_node_versions()

        self.prefix = prefix

        # Guess two things at once:
        self.is_venv = self._guess_version_by_prefix()

        # Our NodeJS interpreters is always in usr/bin on both CL6/CL7, but
        # this code is also called for virtual environments that use
        # different path
        if self.is_venv:
            folder_ = 'bin'
        else:
            folder_ = 'usr/bin'

        self.binary = os.path.join(self.prefix, folder_, 'node')
        if not os.path.isfile(self.binary):
            raise ClSelectExcept.InterpreterError(
                'Cannot find node binary: %s' % self.binary)

        self.version_full = versions_info[self.version]['full_version']

    def _guess_version_by_prefix(self):
        # Example prefixes:
        # /opt/alt/alt-nodejs6/root
        # /home/<user>/nodevenv/<app_root>/6
        if self.prefix.startswith('/opt/alt'):
            venv = False
            pattern = r'/opt/alt/alt-nodejs(\d+)/root'
        else:
            venv = True
            pattern = r'.+/nodevenv/.+/(\d+)'
        self.version = re.findall(pattern, self.prefix)[0]
        return venv

    def __repr__(self):
        return (
            "%s.%s(prefix='%s', version='%s', binary='%s', "
            "version_full='%s')" % (
                self.__class__.__module__, self.__class__.__name__,
                self.prefix, self.version, self.binary, self.version_full
            ))

    def as_dict(self, key=None):
        i = {
            'prefix': self.prefix,
            'binary': self.binary,
            'version': self.version,
            'version_full': self.version_full,
            }
        if key:
            del i[key]
            return {getattr(self, key): i}
        return i


def interpreters(key=None):
    interps = []

    from clselect.clselectnodejs import scan_node_versions
    versions_info = scan_node_versions()

    for ver_info in versions_info.values():
        try:
            interpreter = Interpreter(ver_info['root_path'], versions_info)
        except ClSelectExcept.InterpreterError:
            continue
        interps.append(interpreter)
    if key:
        return dict((getattr(i, key), i) for i in interps) 
    else:
        return interps


def interpreters_dict(key):
    return dict(list(i.as_dict(key).items())[0] for i in interpreters())
