# coding: utf-8

# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

from __future__ import print_function
from __future__ import division
from __future__ import absolute_import
import re

from clselect.baseclselect.pkgmanager import BasePkgManager
from . import (
    CONFIG_DIR,
    is_major_version,
    create_config_dirs,
    ALT_NAMES,
    scan_node_versions,
    ALT_NODE_PREFIX)


class PkgManager(BasePkgManager):
    """
    Class responsible for all interactions with Yum, NodeJS version
    installation/removal and gathering info about already installed versions
    """

    def __init__(self):
        super(PkgManager, self).__init__()

    _config_dir = CONFIG_DIR

    # We should disable all repos but CL's, because Yum will not list anything
    # if some repo is broken

    # TODO We'll improve "Refresh UI button" and implement respective
    # command/option to reset both yum cache and ours in a separate task.
    # {yum_cmd} clean all --disablerepo='*' --enablerepo='cloudlinux-*';

    # https://unix.stackexchange.com/questions/4578/yum-update-error-cant-start-new-thread
    _yum_cmd = 'LANG=C yum --disableplugin=fastestmirror '

    _alt_names = ALT_NAMES

    _redirect_log = '&>/var/log/cl-nodejs-last-yum.log'

    _install_cmd = ''.join([
        _yum_cmd,
        '-y groupinstall ',
        _alt_names, '{} ',
        _redirect_log,
    ])

    # To delete NodeJS versions installed earlier from testing we HAVE TO enable
    # testing repo or yum will not see this groups
    _remove_cmd = ''.join([
        _yum_cmd,
        '--enablerepo=cloudlinux-updates-testing -y groupremove ',
        _alt_names, '{} ',
        _redirect_log,
    ])

    def _get_lock_file_path(self, version):
        return ALT_NODE_PREFIX + '{}/.lock'.format(version)

    def _verify_action(self, version):
        """Do some common pre-installation/uninstallation checks"""
        if not is_major_version(version):
            return 'Invalid version "{}". ' \
                   'It should be positive integer'.format(version)
        working_error = self._check_yum_in_progress()
        if working_error:
            return working_error

    def checkout_available(self):
        """
        Should return list of major versions available to install from
        currently enabled repos.
            Note, this can be an empty list if no NodeJS version has been
            released to repos yet or in case of network/repos/yum problem.
        OR
            None if our cache is updating right now because it was
            absent/outdated/corrupted/etc.
        :rtype: list | None
        """
        # Data example:
        # alt-nodejs6\n
        # alt-nodejs8\n
        # alt-nodejs9\n
        data = self._read_yum_cache()
        if not data:    # No cache, broken cache, outdated cache etc.
            self.update_yum_cache()
            return None

        available = re.findall(r'alt-nodejs(\d+)\n', data)
        return available

    def _scan_interpreter_versions(self):
        return scan_node_versions()

    def _create_config_dirs(self):
        create_config_dirs()

    @staticmethod
    def _is_version_in_use(version):
        # type: (str) -> bool
        """
        Check what passed version isn't used any web-application
        """
        from clselect.clselectnodejs.apps_manager import ApplicationsManager
        apps_manager = ApplicationsManager()
        return apps_manager.is_version_in_use(version)