# coding: utf-8

from __future__ import print_function
from __future__ import division
from __future__ import absolute_import
import os
import re
import glob
import errno

from clselect.baseclselect import BaseSelectorError

# Used both for config and some state like available_versions cache
# mainly because it's easier to put it in cagefs as a one dir
# We have symlink to that dir in /etc
# for ease of use the selector config and avoiding problem with cagefs
CONFIG_DIR = '/usr/share/l.v.e-manager/cl.nodejs'

ALT_NAMES = 'alt-nodejs'
# It's also used to construct base_dir:
ALT_NODE_PREFIX = '/opt/alt/{}'.format(ALT_NAMES)

# This file is specifically created during package build and contains
# full version in the tail of a name
NODE_VERSION_GLOB = '/root/usr/share/node/node-version-'


class NodeJSError(BaseSelectorError):
    """Top level error class for admin's part of NodeJS selector"""
    pass


class NodeJSConfigError(NodeJSError):
    """Generic error class for NodeConfig"""
    pass


def is_major_version(ver):
    """Return True if specified MAJOR version is valid for processing"""
    if not isinstance(ver, str):
        return False
    if not re.match(r'^[1-9][0-9]*$', ver):
        return False
    return True


def create_config_dirs():
    try:
        os.makedirs(CONFIG_DIR)
    except OSError as e:
        if e.errno != errno.EEXIST:   # ignore only "File exists" error
            raise


def scan_node_versions():
    """ Search CL NodeJS interpreters and return dict with info about them """
    res = {}
    for folder in glob.glob('{}*{}*'.format(ALT_NODE_PREFIX,
                                            NODE_VERSION_GLOB)):
        full_ver = folder.split(NODE_VERSION_GLOB)[1]
        if not full_ver:
            # Only when package is broken
            continue

        maj_ver = str(int(full_ver.split('.')[0]))
        res[maj_ver] = {
            'full_version': full_ver,
            'root_path': '{}{}/root'.format(ALT_NODE_PREFIX, maj_ver),
        }
    return res
