# -*- coding: utf-8 -*-

# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT


from __future__ import print_function
from __future__ import division
from __future__ import absolute_import

import traceback

from clcommon import FormattedException
from clcommon.cpapi import (
    getCPName,
    PLESK_NAME,
    CPANEL_NAME,
    DIRECTADMIN_NAME
)


class BaseClSelectException(FormattedException):
    """Base class for all cl-select exceptions"""

    def __init__(self, message, details=None, context=None):
        super(BaseClSelectException, self).__init__(
            dict(message=message, details=details, context=context or {}))


class ClSelectExcept(object):
    """
    Custom exceptions class
    """

    class ConfigNotFound(BaseClSelectException):
        def __init__(self, filename, message=None):
            if message is None:
                message = "Config file (%s) not found" % (filename,)
            super(ClSelectExcept.ConfigNotFound, self).__init__(message)

    class UnableToLoadData(BaseClSelectException):
        def __init__(self, filename, reason):
            message = "Could not load data from %s: %s" % (filename, reason)
            super(ClSelectExcept.UnableToLoadData, self).__init__(message)

    class BusyApplicationRoot(BaseClSelectException):
        def __init__(self, application_root):
            message = 'The application cannot be located inside of already ' \
                      'existing one: {}. Please, remove the already existing ' \
                      'application or specify the other \'application root\' path.'.format(application_root)
            super(ClSelectExcept.BusyApplicationRoot, self).__init__(message)

    class WrongData(BaseClSelectException):
        def __init__(self, message=None, context=None):
            if not message:
                message = "Error in data supplied"
            super(ClSelectExcept.WrongData, self).__init__(message, context=context)

    class WrongConfigFormat(BaseClSelectException):
        def __init__(self, filename):
            message = "Config file (%s) has errors" % filename
            super(ClSelectExcept.WrongConfigFormat, self).__init__(message)

    class NoSuchAlternativeVersion(BaseClSelectException):
        def __init__(self, version):
            message = 'No such alternative version (%s)' % version
            super(ClSelectExcept.NoSuchAlternativeVersion, self).__init__(message)

    class EnvironmentAlreadyExists(BaseClSelectException):
        def __init__(self, path):
            message = 'Virtual environment already exists (%s). ' \
                      'Probably it is used by another application. ' \
                      'Please, remove directory if it is no longer needed.' % path
            super(ClSelectExcept.EnvironmentAlreadyExists, self).__init__(message)

    class AppRootBusy(BaseClSelectException):
        def __init__(self, path):
            message = 'Specified directory already used by \'%s\'' % path
            super(ClSelectExcept.AppRootBusy, self).__init__(message)

    class UnableToProcessOption(BaseClSelectException):
        def __init__(self, data):
            message = 'Cannot process option (%s)' % data
            super(ClSelectExcept.UnableToProcessOption, self).__init__(message)

    class UnableToSaveData(BaseClSelectException):
        def __init__(self, filename, message):
            message = "Unable to save data to %s (%s)" % (filename, message)
            super(ClSelectExcept.UnableToSaveData, self).__init__(message)

    class ExternalProgramFailed(BaseClSelectException):
        def __init__(self, message, **kwargs):
            super(ClSelectExcept.ExternalProgramFailed, self).__init__(message, **kwargs)

    class NoSuchUser(BaseClSelectException):
        def __init__(self, user):
            message = "No such user (%s)" % user
            super(ClSelectExcept.NoSuchUser, self).__init__(message)

    class NoSuchExtension(BaseClSelectException):
        def __init__(self, ext_list, all_extensions):
            extensions = ', '.join(ext_list)
            extensions_aval = ', '.join(all_extensions)
            message = "No such extension(s): %s; available extensions is: %s" % (extensions,
                                                                                 extensions_aval)
            super(ClSelectExcept.NoSuchExtension, self).__init__(message)

    class NoUserSelector(BaseClSelectException):
        def __init__(self, user):
            message = "No selector dir for user %s" % user
            super(ClSelectExcept.NoUserSelector, self).__init__(message)

    class UnableToSetAlternative(BaseClSelectException):
        def __init__(self, user, version, message):
            message = "Unable to set version %s for user %s: (%s)" % (version, user, message)
            super(ClSelectExcept.UnableToSetAlternative, self).__init__(message)

    class UnableToChangeToAnotherUser(BaseClSelectException):
        def __init__(self, user, message):
            message = "Unable to change to user %s (%s)" % (user, message)
            super(ClSelectExcept.UnableToChangeToAnotherUser, self).__init__(message)

    class NotCageFSUser(BaseClSelectException):
        def __init__(self, user):
            message = "User %s not in CageFS" % user
            super(ClSelectExcept.NotCageFSUser, self).__init__(message)

    class UnableToGetExtensions(BaseClSelectException):
        def __init__(self, version):
            message = 'Cannot get extensions list for version "%(version)s". ' \
                      'Please, make sure that native php works fine and run ' \
                      '"%(setup_command)s" to configure PHP selector.'
            super(ClSelectExcept.UnableToGetExtensions, self).__init__(message, context={
                'version': version,
                'setup_command': "cloudlinux-selector setup --interpreter=php --json"
            })

    class NativeNotInstalled(BaseClSelectException):
        def __init__(self, native_php_path):
            cp_name = getCPName()
            message = 'Native php version is not installed ("%(native_php_path)s" binary is missing), ' \
                      'thus CloudLinux PHP selector is not able to work. ' \
                      'Please, install native php and run "%(setup_command)s" to configure PHP selector.'
            context = dict(
                control_panel=cp_name,
                native_php_path=native_php_path,
                setup_command="cloudlinux-selector setup --interpreter=php --json"
            )
            if getCPName() == PLESK_NAME:
                context['http_docs'] = 'https://support.plesk.com/hc/en-us/articles/115004177974-' \
                                       'How-to-install-PHP-by-OS-vendor-in-Plesk-for-Linux'
                message = f'{message} PHP installation instructions for %(control_panel)s: %(http_docs)s'
            elif getCPName() == CPANEL_NAME:
                context['http_docs'] = 'https://documentation.cpanel.net/display/EA4/EasyApache+4+and+' \
                                       'the+ea-php-cli+Package'
                message = f'{message} For %(control_panel)s you should install `ea-php-cli` package: %(http_docs)s'
            elif getCPName() == DIRECTADMIN_NAME:
                context['http_docs'] = 'https://help.directadmin.com/item.php?id=345'
                message = f'{message} For %(control_panel)s you follow this instruction ' \
                          f'to build native php: %(http_docs)s'

            super(ClSelectExcept.NativeNotInstalled, self).__init__(
                message, context=context)

    class MissingCagefsPackage(BaseClSelectException):
        def __init__(self):
            link = "http://docs.cloudlinux.com/index.html?cagefs_installation.html"
            super(ClSelectExcept.MissingCagefsPackage, self).__init__(
                message="Required package `%(package)s` is not installed or broken. "
                        "Please install or reinstall it. Details: %(link)s.",
                context=dict(
                    link=link,
                    package='cagefs'
                )
            )

    class SelectorException(BaseClSelectException):
        def __init__(self, message):
            super(ClSelectExcept.SelectorException, self).__init__(message)

    class InterpreterError(BaseClSelectException):
        def __init__(self, message, context=None):
            super(ClSelectExcept.InterpreterError, self).__init__(
                message, context=context or {})

    class WebAppError(BaseClSelectException):
        def __init__(self, message):
            super(ClSelectExcept.WebAppError, self).__init__(message)

    class EmptyParam(BaseClSelectException):
        def __init__(self, message):
            message = "%s not specified" % message
            super(ClSelectExcept.EmptyParam, self).__init__(message)

    class NoSuchApplication(BaseClSelectException):
        def __init__(self, message):
            super(ClSelectExcept.NoSuchApplication, self).__init__(message)

    class FileProcessError(BaseClSelectException):
        def __init__(self, filename, message=None):
            message = "File (%s): %s" % (filename, message)
            super(ClSelectExcept.FileProcessError, self).__init__(message)

    class CreateSymlinkError(BaseClSelectException):
        def __init__(self, message):
            super(ClSelectExcept.CreateSymlinkError, self).__init__(message)

    class RemoveSymlinkError(BaseClSelectException):
        def __init__(self, message):
            super(ClSelectExcept.RemoveSymlinkError, self).__init__(message)

    class SymlinkError(BaseClSelectException):
        def __init__(self, message):
            super(ClSelectExcept.SymlinkError, self).__init__(message)

    class MissingApprootDirectory(BaseClSelectException):
        def __init__(self, message):
            super(ClSelectExcept.MissingApprootDirectory, self).__init__(message)

    class ConfigMissingError(BaseClSelectException):
        def __init__(self, message):
            super(ClSelectExcept.ConfigMissingError, self).__init__(message)

    class VersionModificationBlocked(BaseClSelectException):
        def __init__(self, message):
            super(ClSelectExcept.VersionModificationBlocked, self).__init__(message)
