# -*- coding: utf-8 -*-

# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT
from __future__ import print_function
from __future__ import division
from __future__ import absolute_import
import json
import sys
import csv
import time

from clcommon.ui_config import UIConfig
from clcommon import FormattedException


def print_text(data, title='id', sort_key=str):
    '''
    Prints data as pure text
    '''
    data_keys = list(data.keys())
    if len(data_keys) == 0:
        return
    fields = list(map((lambda x: x[0]), data[data_keys[0]]))
    fmt_list = ['%32s']
    fmt_list.extend(['%16s'] * (len(fields)))
    fmt = ''.join(fmt_list)
    print(fmt % tuple([title] + fields))
    print('-' * 80)
    for item in sorted(data_keys, key=sort_key):
        row = [item]
        row.extend(list(map((lambda x:x[1]), data[item])))
        print(fmt % tuple(row))


def print_csv(data, title='id', sort_key=str):
    '''
    Prints data as comma separated values
    '''
    data_keys = list(data.keys())
    if len(data_keys) == 0:
        return
    fields = list(map((lambda x: x[0]), data[data_keys[0]]))
    fields = [title] + fields
    csv_out = csv.writer(sys.stdout, quoting=csv.QUOTE_MINIMAL)
    fmt = ','.join(['%s'] * len(fields))
    print(fmt % tuple(fields))
    for item in sorted(data_keys, key=sort_key):
        row = [item]
        row.extend(list(map((lambda x:x[1]), data[item])))
        csv_out.writerow(row)


def print_json(data, title='id', sort_key=str):
    """
    Prints data as json
    """
    print(json.dumps({
        'result': 'success',
        'timestamp': time.time(),
        'items': data,
        'quota_show': check_quota_show(),
    }, indent=1))


def print_json_error(error):
    """
    Prints error in json format that
    is later used by cl-limits
    """
    if isinstance(error, FormattedException):
        print(json.dumps({
            'result': error.message,
            'context': error.context,
            'details': error.details
        }, indent=1))
    else:
        print(json.dumps({
            'result': str(error)
        }, indent=1))


def print_text_error(message):
    print("ERROR: %s" % (message))
    print(
        "Something went wrong while running cl-quota. "
        "Make sure the utility is installed and working correctly. "
        "If the problem persists, contact CloudLinux support for help.",
        end="",
    )


def print_csv_error(message):
    csv_out = csv.writer(sys.stdout, quoting=csv.QUOTE_ALL)
    csv_out.writerow(["ERROR", message])


def check_quota_show():
    """
    Show quota enabled
    :return: bool
    """
    return UIConfig().get_param('showUserInodesUsage', 'inodeLimits') is True
