#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

import glob
import logging
import os.path
from dataclasses import dataclass

import yaml

from clflags.config import DEFAULT_FLAGS_STORAGE_PATH, DEFAULT_META_STORAGE_PATH, DEFAULT_FLAG_GLOB


@dataclass
class FlagInfo:
    name: str
    description: str
    path: str

    @property
    def enabled(self):
        return os.path.isfile(self.path)

    def __hash__(self):
        return hash(self.name)


def list_flags_info() -> list[FlagInfo]:
    """
    Gathers information about flags from DEFAULT_META_STORAGE_PATH
    and returns them in a python-friendly structure.
    """
    meta_files: list[FlagInfo] = []
    storage_glob = os.path.join(DEFAULT_META_STORAGE_PATH, DEFAULT_FLAG_GLOB)

    for meta_file in glob.glob(storage_glob):
        try:
            with open(meta_file, encoding='utf-8') as f:
                metadata = yaml.safe_load(f)

            if metadata is None:
                raise ValueError("Empty or invalid YAML content")

            description = metadata.get('description')
            if not description:
                raise KeyError("Missing 'description' in metadata")

            flag_name = os.path.basename(meta_file)
            default_flag_path = os.path.join(DEFAULT_FLAGS_STORAGE_PATH, flag_name)

            meta_files.append(FlagInfo(
                name=flag_name.rsplit('.')[-2],
                description=description,
                path=metadata.get('path', default_flag_path)
            ))

        except Exception:
            logging.exception("Error processing %s", meta_file)
            logging.error("Skipping flag %s due to an error.", meta_file)
            continue

    return meta_files
