# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2018 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# https://cloudlinux.com/docs/LICENCE.TXT
#
from clcommon.lib import cledition
from clcommon.utils import is_ubuntu
from clconfigure import task
from clconfigure.awp import enable_accelerate_wp_free
from clconfigure.clpro import trigger_manage_clplus
from clconfigure.files import remove_cl_edition_file, update_cl_edition_file
from clconfigure.limits import (
    apply_workaround_lve_failed,
    disable_lve_services,
    unload_lve_module,
)
from clconfigure.linksafe import initialize_linksafe
from clconfigure.packages import (
    STATE_INSTALLED,
    STATE_REMOVED,
    erase_package,
    get_package_state,
    install_package,
    set_package_state,
)
from clconfigure.services import (
    STATE_RUNNING,
    STATE_STOPPED,
    set_service_state,
)

# TODO: needs to be elaborated
CONFLICTED_PACKAGES = (
    'mod_hostinglimits',
    'ea-apache24-mod_hostinglimits',
    'httpd24-mod_hostinglimits',
    'governor-mysql',
)
if is_ubuntu():
    # Replace underscores with hyphens, since on Ubuntu,
    # underscores are prohibited in package names
    CONFLICTED_PACKAGES = tuple(pkg.replace('_', '-') for pkg in CONFLICTED_PACKAGES)


@task(f"Initial provisioning during transition of {cledition.get_cl_edition_readable()} into CloudLinux OS Solo")
def preprovision(edition):
    """
    Triggered before jwt token change in rhn-client-tools
    but already after successful registration with new edition.

    Designed to safely terminate things that should not be
    running as part of current edition but avoid license checks.
    """


@task("Configuring environment for CloudLinux OS Solo")
def provision(edition):
    """
    Converts "normal" CloudLinux into CL Solo by disabling
    some services and installing some packages.
    """
    remove_cl_edition_file()
    erase_package('cloudlinux-admin-meta')
    for conflicted_package in CONFLICTED_PACKAGES:
        set_package_state(STATE_REMOVED, conflicted_package)
    set_package_state(STATE_INSTALLED, 'cloudlinux-solo-meta')
    update_cl_edition_file()

    disable_lve_services()
    initialize_linksafe()
    # Reinstall lve-utils to set up correct crons
    install_package('lve-utils', reinstall=True)

    if get_package_state('cagefs') == STATE_INSTALLED:
        install_package('cagefs', reinstall=True)
        # Restart cagefs service to create namespaces
        set_service_state(STATE_STOPPED, 'cagefs')
        set_service_state(STATE_RUNNING, 'cagefs')

    enable_accelerate_wp_free()
    trigger_manage_clplus()
    unload_lve_module()
    # When lve service is stopped with loaded kmodlve in `disable_lve_services`, it can go to failed state
    # Try to fix it and to move to `inactive` state
    apply_workaround_lve_failed()
