# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2018 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# https://cloudlinux.com/docs/LICENCE.TXT
#
from concurrent.futures import ThreadPoolExecutor
from typing import Iterator

from clcommon.utils import is_ubuntu
from clconfigure import task, run
from clconfigure.packages import install_package

# Files of the packages listed below are not supposed
# to allow creation of symlinks to them
PACKAGES_TO_EXCLUDE = {
    'alt-python27-cllib',
    'alt-php-xray',
    'alt-php-ssa',
}


def initialize_linksafe():
    # NOTE(vlebedev): Also we need to trigger linksafe scriptlets to properly initialize relevant UNIX groups.
    #                 Otherwise this problem may occur:
    #                 https://cloudlinux.zendesk.com/hc/en-us/articles/
    #                 115004139173-What-does-Error-failed-to-create-symlink-13-Permission-denied-mean
    install_package('cloudlinux-linksafe', reinstall=True)
    assign_linksafe_group()


@task("Assigning linksafe group to files of 'alt-*' packages residing under /opt/alt")
def assign_linksafe_group(concurrency: int = 5):
    with ThreadPoolExecutor(max_workers=concurrency) as tp:
        for package in _installed_packages_iter(
            "alt-php*",
            "alt-python*",
            "alt-ruby*",
        ):
            if package not in PACKAGES_TO_EXCLUDE:
                tp.submit(run, ['/usr/bin/cl-linksafe-apply-group', package])


def _installed_packages_iter(*patterns: str) -> Iterator[str]:
    if is_ubuntu():
        result = run(["dpkg-query", "-W", "-f", "${Package}\n", *patterns])
    else:
        result = run(["rpm", "-qa", *patterns])

    lines = result.stdout.strip().splitlines()
    for line in lines:
        yield line
