# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2018 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# https://cloudlinux.com/docs/LICENCE.TXT
#
import logging
from clconfigure import task, run

STATE_MASKED = 'masked'
STATE_STOPPED = 'stopped'

STATE_UNMASKED = 'unmasked'
STATE_RUNNING = 'running'

STATE_FAILED = 'failed'

STATUS_STOPPED = 'inactive'
STATUS_RUNNING = 'active'

STATE_ENABLED = 'enabled'
STATE_DISABLED = 'disabled'


def get_service_state(service, kind='is-active'):
    """
    Get information about systemctl service state
    (either activity or enability)
    """
    result = run(['systemctl', kind, service + '.service'])

    state = result.stdout.strip()
    return state


@task("Changing service '{service_name}' state to '{desired_state}'")
def set_service_state(desired_state, service_name):
    """
    Brings service to given state. May be executed more that once,
    does't crash on future calls
    """
    current_state = get_service_state(service_name, kind='is-enabled')
    current_status = get_service_state(service_name)

    logging.debug("Checking service '%s' state... package is in state='%s' running='%s'"
                  "", service_name, current_state, current_status)

    action = None
    if desired_state == STATE_MASKED and current_state != STATE_MASKED:
        action = 'mask'
    elif desired_state == STATE_UNMASKED and current_state == STATE_MASKED:
        action = 'unmask'
    elif desired_state == STATE_STOPPED and current_status != STATUS_STOPPED:
        action = 'stop'
    elif desired_state == STATE_DISABLED and current_state != STATE_DISABLED:
        action = 'disable'
    elif desired_state == STATE_RUNNING and current_status != STATUS_RUNNING:
        action = 'start'
    elif desired_state == STATE_ENABLED and current_state != STATE_ENABLED:
        action = 'enable'

    if action is None:
        logging.debug("No actions needed for service '%s'", service_name)
    else:
        run(['systemctl', action, service_name + '.service'])
        current_state = get_service_state(service_name)
        current_status = get_service_state(service_name)
        logging.debug("Checking service '%s' state again... service now is in state='%s' running='%s'"
                      "", service_name, current_state, current_status)
