# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2024 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# https://cloudlinux.com/docs/LICENCE.TXT
#
import functools
import logging
import os

from clcommon.const import Feature
from clcommon.cpapi import is_panel_feature_supported
from clconfigure import run, task
from clconfigure.packages import install_package

CAGEFSCTL = "/usr/sbin/cagefsctl"


def _skip_without_cagefs(func):
    @functools.wraps(func)
    def decorator(*args, **kwargs):
        if not is_panel_feature_supported(Feature.CAGEFS):
            logging.debug("Feature 'CageFS' is not supported. Skipping...")
            return

        if not os.path.exists(CAGEFSCTL):
            logging.debug("CageFS is not installed. Skipping...")
            return

        return func(*args, **kwargs)
    return decorator


@_skip_without_cagefs
@task("Cleaning CageFS environment")
def clean_cagefs_without_lve_environment():
    if is_panel_feature_supported(Feature.LVE):
        logging.debug("LVE feature is supported. Skipping...")
        return

    result = run([CAGEFSCTL, "--clean-without-lve-environment"])
    if result.exitcode != 0:
        logging.warning("Unable to clean non-LVE CageFS enviroment. stdout: %s, stderr: %s",
                        result.stdout, result.stderr)


@_skip_without_cagefs
@task("Reconfiguring CageFS")
def reconfigure_cagefs():
    install_package("cagefs", reinstall=True)
    _rebuild_apache_directadmin()


def _rebuild_apache_directadmin():
    da_bin = "/usr/local/directadmin/directadmin"
    if not os.path.exists(da_bin):
        return

    result = run([da_bin, "build", "apache"])
    if result.exitcode != 0:
        logging.warning("Error during rebuilding Apache. stdout: %s, stderr: %s",
                        result.stdout, result.stderr)
