#!/opt/cloudlinux/venv/bin/python -bb
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2018 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# https://cloudlinux.com/docs/LICENCE.TXT
#
import argparse
import enum
import logging
import sys

from clcommon.lib import cledition
from clconfigure import setup_logger
from clconfigure.spec import admin, shared, shared_pro, solo
from lve_utils.sentry import init_lve_utils_sentry_client

LOGS_PATH = '/var/log/cloudlinux/clcustomizer.log'


class TargetEdition(enum.Enum):
    ADMIN = 'admin'
    AUTO = 'auto'
    SHARED = 'shared'
    SHARED_PRO = 'shared_pro'
    SOLO = 'solo'


def _resolve_auto_target():
    """
    Resolves edition for 'auto' target.
    """
    if cledition.is_cl_solo_edition():
        return TargetEdition.SOLO.value
    if cledition.is_cl_admin_edition():
        return TargetEdition.ADMIN.value
    if cledition.is_cl_shared_edition():
        return TargetEdition.SHARED.value
    if cledition.is_cl_shared_pro_edition():
        return TargetEdition.SHARED_PRO.value
    raise NotImplementedError()


def main():
    """
    Entry point for this program.
    Parses arguments and calls needed methods.
    """
    setup_logger(None, LOGS_PATH)
    logging.debug('Executing "%s"', ' '.join(sys.argv))

    parser = argparse.ArgumentParser()
    subparsers = parser.add_subparsers(dest='action')
    reconfigure = subparsers.add_parser('reconfigure')
    reconfigure.add_argument('-t', '--target',
                             choices=[e.value for e in TargetEdition],
                             required=True)

    preconfigure = subparsers.add_parser('preconfigure')
    preconfigure.add_argument('-t', '--target',
                              choices=[e.value for e in TargetEdition],
                              required=True)

    args = parser.parse_args()
    if args.action is None:
        parser.print_help()
        return

    target = _resolve_auto_target() if args.target == 'auto' else args.target

    handlers = {
        TargetEdition.SOLO.value: solo,
        TargetEdition.ADMIN.value: admin,
        TargetEdition.SHARED.value: shared,
        TargetEdition.SHARED_PRO.value: shared_pro,
    }

    handler = handlers.get(target)
    if handler is None:
        raise ValueError(f'Unknown target {target}')

    actions = {
        'reconfigure': 'provision',
        'preconfigure': 'preprovision'
    }
    action = actions.get(args.action)
    if action is None:
        raise ValueError(f"Unknown action {args.action}")

    getattr(handler, action)(edition=target)


if __name__ == '__main__':
    logging.basicConfig(level=logging.INFO)

    init_lve_utils_sentry_client('clconfigure')
    try:
        main()
    finally:
        logging.info('You can find complete logs of the operation here: %s', LOGS_PATH)
