# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2018 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# https://cloudlinux.com/docs/LICENCE.TXT
#
import logging
import os
import re

from clcommon.utils import is_ubuntu
from clconfigure import task, run

STATE_REMOVED = 'uninstalled'
STATE_INSTALLED = 'installed'


@task("Erasing package '{package_name}'")
def erase_package(package_name):
    if is_ubuntu():
        run(['dpkg', '--purge', '--force-depends', package_name])
    else:
        run(['rpm', '--erase', '--nodeps', package_name])

    current_state = get_package_state(package_name)

    logging.info("Checking package '%s' state again... package is now in state '%s'", package_name, current_state)
    if STATE_REMOVED != current_state:
        raise RuntimeError("Failed to do required actions")


@task("Changing package '{package_name}' state to '{desired_state}'")
def set_package_state(desired_state, package_name):
    """
    Brings package to given state (installed | uninstalled).
    May be executed more than once, doesn't crash on future calls
    """
    current_state = get_package_state(package_name)
    logging.debug("Checking package '%s' state... package is in state '%s'", package_name, current_state)

    if desired_state == current_state:
        logging.debug("No actions needed for package '%s'", package_name)
        return

    logging.info('State does not match target, changing...')
    if desired_state == STATE_REMOVED:
        if is_ubuntu():
            run(['apt-get', 'purge', '-y', package_name])
        else:
            run(['yum', 'remove', '-y', package_name])
    elif desired_state == STATE_INSTALLED:
        if is_ubuntu():
            run(['apt-get', 'install', '-y', package_name])
        else:
            run(['yum', 'install', '-y', package_name])
    else:
        raise NotImplementedError()

    current_state = get_package_state(package_name)
    logging.info("Checking package '%s' state again... package is now in state '%s'", package_name, current_state)
    if desired_state != current_state:
        raise RuntimeError("Failed to do required actions")


@task("(Re)installing package '{package_name}'")
def install_package(package_name: str, reinstall: bool = False):
    state = get_package_state(package_name)
    if state == STATE_INSTALLED and reinstall:
        cmd = ['apt-get' if is_ubuntu() else 'yum', 'reinstall', '-y', package_name]
    elif state == STATE_REMOVED:
        cmd = ['apt-get' if is_ubuntu() else 'yum', 'install', '-y', package_name]
    else:
        return
    run(cmd)


def get_package_state(package):
    """
    Gets current package state.
    Either installed or uninstalled.
    """
    if is_ubuntu():
        resp = run(['dpkg', '-s', package])
        if resp.exitcode == 0 and 'Status: install ok installed' in resp.stdout:
            return STATE_INSTALLED
        return STATE_REMOVED

    resp = run(['rpm', '-qv', package])
    return STATE_INSTALLED if resp.exitcode == 0 else STATE_REMOVED


def get_list_installed_alt_phps():
    """
    Gets installed alt-phps
    return: list ['php44', 'php54', 'php80']
    """
    return [php for php in os.listdir('/opt/alt') if re.match(r'^php\d+$', php)]
