# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2018 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# https://cloudlinux.com/docs/LICENCE.TXT
#
from clcommon.lib import cledition

from clconfigure import task
from clconfigure.awp import enable_accelerate_wp_free
from clconfigure.cagefs import clean_cagefs_without_lve_environment, reconfigure_cagefs
from clconfigure.clpro import trigger_manage_clplus
from clconfigure.files import update_cl_edition_file, remove_cl_edition_file
from clconfigure.limits import (
    initialize_lvestats,
    set_default_limits_state,
    disable_lve_services
)
from clconfigure.linksafe import initialize_linksafe
from clconfigure.packages import (
    STATE_REMOVED,
    set_package_state,
    STATE_INSTALLED,
    erase_package,
    install_package
)


@task(f"Initial provisioning during transition of {cledition.get_cl_edition_readable()} into CloudLinux OS Admin")
def preprovision(edition):
    """
    Triggered before jwt token change in rhn-client-tools
    but already after successful registration with new edition.

    Designed to safely terminate things that should not be
    running as part of current edition but avoid license checks.
    """
    clean_cagefs_without_lve_environment()


@task("Configuring environment for CloudLinux OS Admin")
def provision(edition):
    """
    Converts "normal" CloudLinux into CL Admin by changing default limits
    and some other stuff that may be added later.
    """
    remove_cl_edition_file()
    erase_package('cloudlinux-solo-meta')

    # Remove governor-mysql but leave patched DBMS packages
    # Currently removing DBMS is not reliable as long as many cases are not worked out well
    # e.g. when upstream DBMS package version is not available in repo
    set_package_state(STATE_REMOVED, 'governor-mysql')

    set_package_state(STATE_INSTALLED, 'cloudlinux-admin-meta')
    update_cl_edition_file()
    enable_accelerate_wp_free()

    if not cledition.is_container():
        initialize_linksafe()
        initialize_lvestats()
        set_default_limits_state('unlimited')
    else:
        disable_lve_services()

    # Reinstall lve-utils:
    # - to apply correct sysctl parameters, see clsupergid_process.py
    # - to fix 'check_symlinkowngid' function in cldiag
    install_package('lve-utils', reinstall=True)
    # Reinstall lvemanager:
    # - to update linksafe group on the wrappers files
    #   within '/usr/share/l.v.e-manager/utils', see 'post' scriptlet
    # - as a workaround for CLOS-2394
    install_package('lvemanager', reinstall=True)
    reconfigure_cagefs()
    trigger_manage_clplus()
