# -*- coding: utf-8 -*-

# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2022 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT
#

import re
import os
from typing import Optional

from clcommon.utils import get_mount_point, run_command, ExternalProgramFailed

USER_QUOTA_ON_PATTERN = re.compile(r'user quota on .+ \(.+\) is on')


def check_quota_enabled(path: str = '/home') -> Optional[str]:
    """
    Check if quotas enabled and initialised for specific path
    :return: string message with giagnostick information if disabled; None if enabled
    """
    if not os.path.isfile('/sbin/quotaon'):
        return 'Package "quota" not installed'

    try:
        mount_point = get_mount_point(path)
    except (OSError, ExternalProgramFailed) as e:
        return f'Error during mount point detection:\n{e}'

    try:
        quotaon_cmd = ['/sbin/quotaon', '-up', mount_point]
        quotaon_output = run_command(quotaon_cmd)
    except ExternalProgramFailed as e:
        quotaon_output = str(e)

    quotaon_output = quotaon_output.strip()
    # detect "user quota on / (/dev/mapper/VolGroup-lv_root) is on"
    if not USER_QUOTA_ON_PATTERN.search(quotaon_output):
        return quotaon_output
    return None
