# -*- coding: utf-8 -*-

# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2018 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT
#

import datetime
import fcntl
import sys

import psutil


class ClAuditLog:
    def __init__(self, log_file_name: str):
        self._log_file_name = log_file_name

    def info_log_write(self, msg: str = ""):
        dt_now = datetime.datetime.now()
        current_proc = psutil.Process()

        # Initialize default values
        parent_proc_name, parent_proc_cmdline = "<Unknown_name>", "<Unknown_cmdline>"
        parent_proc_pid = current_proc.ppid()

        # Attempt to get the parent process properties
        parent_proc = current_proc.parent()
        if parent_proc is not None:
            with parent_proc.oneshot():
                parent_proc_name = parent_proc.name()
                parent_proc_cmdline = " ".join(parent_proc.cmdline())

        try:
            with open(self._log_file_name, "a", encoding="utf-8") as f:
                fcntl.lockf(f.fileno(), fcntl.LOCK_EX | fcntl.LOCK_NB)
                f.write(
                    f"{dt_now.strftime('%Y.%m.%d %H:%M:%S')}: "
                    f"Parent process: {parent_proc_name} ({parent_proc_cmdline}) "
                    f"(Parent PID: {parent_proc_pid}): Args: {sys.argv[1:]} {msg}\n"
                )
        except (IOError, OSError):
            # Ignore all file errors
            pass
