﻿# -*- coding: utf-8 -*-

# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2018 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT
#
import os
import sys
import subprocess

class ExternalProgramFailed(Exception):
    """Exception class for errors related to externally executed commands."""
    def __init__(self, message: str = ""):
        Exception.__init__(self, message)


def check_command(cmdname):
    """
    Checks if command is present and exits if no
    """
    if not os.path.exists(cmdname):
        print(f"Command not found: ({cmdname})")
        sys.exit(1)


def run_command(cmd, env_data=None, return_full_output=False, std_in=None, convert_to_str=True):
    """
    Runs external process and returns output
    :param cmd: command and arguments as a list
    :param env_data: environment data for process
    :param return_full_output: if true, returns (ret_code, std_out, std_err)
    @return: process stdout if is_full_output==False
    else - cortege (ret_code, std_out, std_err) without any checking
    """
    cmd_line = " ".join(cmd)
    try:
        std_err_obj = subprocess.PIPE if return_full_output else subprocess.STDOUT
        stdin_arg = subprocess.PIPE if std_in else subprocess.DEVNULL
        with subprocess.Popen(
            cmd,
            stdin=stdin_arg,
            stdout=subprocess.PIPE,
            stderr=std_err_obj,
            close_fds=True,
            env=env_data,
            text=convert_to_str,
        ) as proc:
            if not std_in:
                std_out, std_err = proc.communicate()
            else:
                std_out, std_err = proc.communicate(std_in)
            returncode = proc.returncode
    except OSError as oserr:
        raise ExternalProgramFailed(
            f"{cmd_line}. Cannot run command: {str(oserr)}"
        ) from oserr

    if return_full_output:
        return returncode, std_out, std_err
    if returncode != 0:
        if not convert_to_str:
            raise ExternalProgramFailed(f"Error during command execution: {cmd_line}")
        else:
            raise ExternalProgramFailed(
                std_err or f"output of the command: {cmd_line}\n{std_out}"
            )
    return std_out


def exec_utility(util_path, params):
    """
    Executes supplied utility with supplied parameters
    :param util_path: Executable file to run path
    :param params: utility parameters
    :return: Cortege (ret_code, utility_stdout)
    """
    args = []
    args.append(util_path)
    args.extend(params)
    with subprocess.Popen(args, stdout=subprocess.PIPE, text=True) as proc:
        stdout, _ = proc.communicate()
        retcode = proc.returncode
    return retcode, stdout.strip()
