# -*- coding: utf-8 -*-
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2022 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT
#

import glob
import importlib.util
import os
import sys

from .helpers import LISTENERS_DIRECTORY

from .modify_domain_hook import ModifyDomainHook
from .modify_admin_hook import ModifyAdminHook
from .modify_user_hook import ModifyUserHook
from .modify_package_hook import ModifyPackageHook


def import_file(filepath):
    # Fast path: see if the module has already been imported.
    name = os.path.basename(filepath)[:-3]
    if name in sys.modules:
        return sys.modules[name]

    # If any of the following calls raises an exception,
    # there's a problem we can't handle -- let the caller handle it.
    module_spec = importlib.util.spec_from_file_location(name, filepath)
    if module_spec is None:
        raise ImportError(f"Can't find module {name} at {filepath}")

    # Load the module, potentially raising ImportError if the module cannot be loaded
    module = importlib.util.module_from_spec(module_spec)
    module_spec.loader.exec_module(module)

    # Add the module to sys.modules
    sys.modules[name] = module
    return module


# automatically scan directory for universal hooks
for hook_py in glob.glob(os.path.join(LISTENERS_DIRECTORY, '*.py')):
    import_file(hook_py)


__all__ = (
    'ModifyDomainHook',
    'ModifyAdminHook',
    'ModifyUserHook',
    'ModifyPackageHook',
)
