# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2020 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENCE.TXT
#

from jwt import decode, exceptions
from clcommon.lib.consts import DEFAULT_JWT_ES_TOKEN_PATH


def read_jwt(jwt_path):
    with open(jwt_path, mode='rb') as f:
        token_string = f.read().strip()
    return token_string


def decode_jwt(token_string, verify_exp=True):
    jwt_token = decode(token_string, algorithms=['HS256'],
                       options={'require_exp': True, "verify_exp": verify_exp, "verify_iss": True,
                                'verify_signature': False},
                       issuer='CloudLinux')
    if 'exp' not in jwt_token:
        raise exceptions.MissingRequiredClaimError('exp')
    return jwt_token


def jwt_token_check():
    """
    JWT token check
    :return: Tuple: success_flag, error_message, token_string
     success_flag: True/False - OK, JWT is valid/Error, invalid
     error_message: Error message
     token_string: Token string
    """
    success_flag, error_message, token_string = True, "OK", None
    try:
        token_string = read_jwt(DEFAULT_JWT_ES_TOKEN_PATH)
    except (OSError, IOError):
        return False, f"JWT file {DEFAULT_JWT_ES_TOKEN_PATH} read error", None
    # JWT read success
    try:
        jwt_token = decode_jwt(token_string)
        if jwt_token.get("cl_plus") is None:
            success_flag, error_message, token_string = False, "JWT token format error", None
        elif not jwt_token.get("cl_plus"):
            success_flag, error_message, token_string = False, "JWT token doesn't have CL+ service", None
    except exceptions.InvalidIssuerError:
        success_flag, error_message, token_string = False, "JWT token issuer is invalid", None
    except exceptions.ExpiredSignatureError:
        success_flag, error_message, token_string = False, "JWT token expired", None
    except exceptions.PyJWTError:
        success_flag, error_message, token_string = False, "JWT token format error", None
    return success_flag, error_message, token_string
