# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2018 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT
import itertools
from enum import StrEnum

from clcommon.lib.cledition import (
    is_cl_admin_edition,
    is_cl_shared_pro_edition_safely,
    is_cl_solo_edition,
    is_container,
)
from clcommon.utils import is_ubuntu


class Feature(StrEnum):
    """Enum of available CloudLinux features
    that vendors can disable or enable.
    """

    PHP_SELECTOR = "php_selector"
    RUBY_SELECTOR = "ruby_selector"
    PYTHON_SELECTOR = "python_selector"
    NODEJS_SELECTOR = "nodejs_selector"
    LSAPI = "mod_lsapi"
    GOVERNOR = "mysql_governor"
    LVE = "lve"
    WIZARD = "wizard"
    CAGEFS = "cagefs"
    RESELLER_LIMITS = "reseller_limits"
    XRAY = "xray"
    WPOS = "accelerate_wp"
    AUTOTRACING = "autotracing"


ALL_CL_FEATURES: list[Feature] = list(Feature)

_UBUNTU_BASED_CL_FEATURES_DISABLED = (
    Feature.RUBY_SELECTOR,
    Feature.PYTHON_SELECTOR,
    Feature.NODEJS_SELECTOR,
    Feature.XRAY,
    Feature.AUTOTRACING,
    Feature.WPOS,
    Feature.WIZARD,
)


def _get_features_disabled_by_system_type() -> tuple:
    """
    Return overrides that are applied to the list of features
    depending on the current system base os.

    Right now we support two backends: rhel-like almalinux and ubuntu.
    """
    if is_ubuntu():
        return _UBUNTU_BASED_CL_FEATURES_DISABLED
    return ()


SOLO_CL_FEATURES_DISABLED = (
    Feature.RUBY_SELECTOR,
    Feature.GOVERNOR,
    Feature.LVE,
    Feature.RESELLER_LIMITS,
)

ADMIN_CL_FEATURES_DISABLED = (
    Feature.RESELLER_LIMITS,
    Feature.GOVERNOR,
)

SHARED_CL_FEATURES_DISABLED = (
    Feature.XRAY,
    Feature.WPOS,
    Feature.AUTOTRACING,
)

SHARED_PRO_CL_FEATURES_DISABLED = ()


def _get_features_disabled_by_license() -> tuple:
    if is_cl_solo_edition(skip_jwt_check=True):
        return SOLO_CL_FEATURES_DISABLED
    elif is_cl_admin_edition(skip_jwt_check=True):
        return ADMIN_CL_FEATURES_DISABLED
    elif is_cl_shared_pro_edition_safely():
        return SHARED_PRO_CL_FEATURES_DISABLED
    else:
        return SHARED_CL_FEATURES_DISABLED


CONTAINER_CL_FEATURES_DISABLED = (
    Feature.RESELLER_LIMITS,
    Feature.GOVERNOR,
    Feature.LVE,
    Feature.RUBY_SELECTOR,
)


def _get_features_disabled_by_environment() -> tuple:
    """
    Return overrides that are applied to the list of features
    depending on the current system environment.

    Right now we support two envs:
        - bare metal or virtualized servers
        - virtuoso containers
    """
    if is_container():
        return CONTAINER_CL_FEATURES_DISABLED
    return ()


def get_control_panel_disabled_features() -> tuple:
    # not the best way, but we have to keep backwards compatibility in this lib
    from clcommon import cpapi  # pylint: disable=cyclic-import,import-outside-toplevel

    return cpapi.get_unsupported_features()


def get_cl_feature_status_map() -> dict[Feature, bool]:
    """
    Return dict of CloudLinux features that can be used
    on current system. The key is the feature name
    and the value is True if feature is allowed to be used
    and False otherwise.

    By default, all features are assumed to be supported
    later we check different conditions and turn off features that
    are not supported on current CloudLinux intallation.
    """
    panel_supported_features = get_control_panel_disabled_features()
    features_disabled_by_system_type = _get_features_disabled_by_system_type()
    features_disabled_by_license = _get_features_disabled_by_license()
    features_disabled_by_environment = _get_features_disabled_by_environment()

    unsuppored_features = set(
        itertools.chain(
            features_disabled_by_system_type,
            features_disabled_by_license,
            features_disabled_by_environment,
            panel_supported_features,
        ),
    )

    # keep feature supported only if all three conditions met:
    #   - feature is supported by current system type (rhel/debian)
    #   - feature is supported by current license type (solo/admin/shared)
    #   - feature is supported by current environment (container/vps)
    #   - feature is supported by current control panel (cpanel/plesk/da)
    return {
        feature: feature not in unsuppored_features
        for feature in ALL_CL_FEATURES
    }


def get_cl_enabled_features() -> list[Feature]:
    """
    Return a list of enabled features.
    """
    return [
        feature
        for feature, is_feature_enabled in get_cl_feature_status_map().items()
        if is_feature_enabled
    ]


MAX_HOSTING_ACCOUNTS_SOLO_EDITION = 1
MAX_HOSTING_ACCOUNTS_ADMIN_EDITION = 5


def get_hosting_accounts_limit() -> int | None:
    """
    Return the maximum number of hosting accounts that
    are currently allowed by CloudLinux license.
    """
    if is_cl_solo_edition(skip_jwt_check=True):
        return MAX_HOSTING_ACCOUNTS_SOLO_EDITION
    elif is_cl_admin_edition(skip_jwt_check=True):
        return MAX_HOSTING_ACCOUNTS_ADMIN_EDITION
    return None
