# -*- coding: utf-8 -*-

# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT
#

import json
import copy
from collections.abc import Mapping

UI_CONFIG_FILE = "/usr/share/l.v.e-manager/lvemanager-config.json"
DEFAULT_UI_CONFIG = {'inodeLimits': {'showUserInodesUsage': False},
                     'uiSettings': {'hideRubyApp': False,
                                    'hideLVEUserStat': False,
                                    'hidePythonApp': False,
                                    'hideNodeJsApp': False,
                                    'hidePHPextensions': False,
                                    'hideDomainsTab': False,
                                    'hidePhpApp': False,
                                    'hideXrayApp': True,
                                    'hideAccelerateWPApp': True,
                                    }
                     }


class UIConfig:
    def __init__(self, ui_config_path=UI_CONFIG_FILE):
        self.ui_config_path = ui_config_path
        self.file_content = {}
        self.config = {}
        try:
            with open(ui_config_path, encoding='utf-8') as json_file:
                self.file_content = json.load(json_file)
                self.config = self.file_content['ui_config']
        except IOError:
            self.config = DEFAULT_UI_CONFIG
        except (ValueError, KeyError):
            self.file_content['ui_config'] = DEFAULT_UI_CONFIG
            self.config = DEFAULT_UI_CONFIG
            self._update_file(ui_config_path)

    def _update_file(self, file_path):
        with open(file_path, 'w', encoding='utf-8') as update_file:
            json.dump(self.file_content, update_file)

    def get_config(self):
        initial_config = copy.deepcopy(DEFAULT_UI_CONFIG)
        self._recursive_assign(initial_config, self.config)
        return initial_config

    def set_config(self, new_config):
        initial_config = copy.deepcopy(self.get_config())
        self._recursive_assign(initial_config, new_config)
        with open(self.ui_config_path, 'w', encoding='utf-8') as json_file:
            json.dump({'ui_config': initial_config}, json_file)

    def get_param(self, param, parent_param=None):
        config = self.get_config()
        if parent_param is None:
            return config.get(param)
        elif config.get(parent_param) is not None:
            return config.get(parent_param).get(param)
        return None

    def _recursive_assign(self, initial_config, new_config):
        """
        Function that recursively assignees values received from the UI
        :param initial_config: Config to which values are assigned
        :param new_config: New config file received from UI
        :return:
        """
        for key, value in new_config.items():
            if isinstance(value, Mapping):
                initial_config[key] = self._recursive_assign(initial_config.get(key, {}), value)
            elif value is not None and initial_config.get(key) is not None:
                initial_config[key] = value
        return initial_config
