# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

from lvestats.lib import lveinfolib_gov
from lvestats.lib.chart import ChartMain
from lvestats.lib.commons import dateutil, sizeutil

VERSION = '0.10'


def _io_units(v):
    """
    Get string representation of value;
    :param int v: value in megabytes
    :return str: humanized value

    >>> _io_units(1)
    '1MB'
    >>> _io_units(1024)
    '1GB'
    """
    return sizeutil.convert_bytes_for_graph(v * 1024 * 1024)


class DbGovChart(ChartMain):
    def __init__(self, config=None):
        super().__init__('dbgovchart',
                         'Creates a chart representing usage pattern for LVE/user',
                         config)

    @staticmethod
    def get_version():
        return VERSION

    def get_chart_data(self, engine, from_ts, to_ts, server, user_id=None, show_all=False):
        utc_from = dateutil.local_to_gm(from_ts)
        utc_to = dateutil.local_to_gm(to_ts)

        dt = utc_to - utc_from
        period_sec = dt.total_seconds()
        show_columns = ('ts', 'cpu', 'lcpu', 'read', 'lread', 'write', 'lwrite')
        data = lveinfolib_gov.HistoryShowDBGov(
            engine,
            utc_from,
            utc_to,
            uid=user_id,
            server_id=server,
            show_columns=show_columns,
            cfg=self.cfg
        ).history_dbgov_show()
        data_collected = DbGovChart.convert_dbdata_to_dict(data, show_columns)
        times = data_collected['ts']
        del data_collected['ts']
        return data_collected, times, period_sec

    def customize_parser(self, parser):
        parser.add_argument('--user',
                            help='mysql username',
                            dest='user_name',
                            required=True)
        return parser

    def add_graphs(self, renderer, data_collected, times, lve_version, show_all, is_user=False):
        if any(data_collected['lcpu']) or show_all:
            if is_user and self.is_normalized_user_cpu:
                # Magnify aCPU, lCPU to 100%
                # New user's data and limits
                a_cpu_new = []
                l_cpu_new = []
                for idx in range(0, len(data_collected['lcpu'])):
                    l_cpu = data_collected['lcpu'][idx]
                    a_cpu = data_collected['cpu'][idx]
                    # New limit
                    l_cpu_new.append(100.0)
                    # New average
                    a_cpu_new.append(a_cpu * 100.0 / l_cpu)
                # Store new data
                data_collected['lcpu'] = l_cpu_new
                data_collected['cpu'] = a_cpu_new
            renderer.add_graph(data_collected, 'CPU Usage',
                               legend={
                                   'cpu': ('CPU', 'green', int),
                                   'lcpu': ('limit', 'red'),
                               }, x_values=times, min_y=0, max_y=100, unit='%', y_legend_converter=int)
        if any(data_collected['lread']) or show_all:
            renderer.add_graph(data_collected, 'Read Usage',
                               legend={
                                   'read': ('READ', 'green',),
                                   'lread': ('limit', 'red'),
                               }, x_values=times, min_y=0, y_legend_converter=_io_units, unit='/s')
        if any(data_collected['lwrite']) or show_all:
            renderer.add_graph(data_collected, 'Write Usage',
                               legend={
                                   'write': ('WRITE', 'green',),
                                   'lwrite': ('limit', 'red'),
                               }, x_values=times, min_y=0, y_legend_converter=_io_units, unit='/s')

        if not show_all and not any(data_collected.values()):
            # if no data available draw text box
            renderer.add_text_box("No data")

        renderer.add_common_x_legend(times, 6)

# Uncomment this if you want to test it.
# if __name__ == "__main__":
#     DbGovChart(config_for_debug={
#         'db_type': 'sqlite',
#         'sqlite_db_path': '/home/shaman/workspace/cloudlinux/dbgovdb/lvestats2.db',
#         'server_id': 'localhost',
#     }).main('--from', '2015-03-12 00:00', '--to', '2015-04-01 00:00', '--user', 'mysqldd', '--output', '/tmp/1.svg')
