Интернационализация
====================

> Note: Этот раздел находится в разработке

Интернационализация (I18N) является частью процесса разработки приложения, которое может быть адаптировано для 
нескольких языков без изменения программной логики. Это особенно важно для веб-приложений, так как потенциальные
пользователи могут приходить из разных стран.

Yii располагает несколькими средствами, призванными помочь с интернационализацией веб-приложения: [переводом 
сообщений](#message-translation), [форматированием чисел и дат](#date-number).

Локализация и языки <span id="locale-language"></span>
------------------------------------------------------

В Yii приложении определены два языка: [[yii\base\Application::$sourceLanguage|исходный язык]] и [[yii\base\Application::$language|язык перевода]].

На "исходном языке" написаны сообщения в коде приложения. Если мы определяем исходным языком английский, то
в коде можно использовать конструкцию:

```php
echo \Yii::t('app', 'I am a message!');
```

Язык перевода определяет, в каком виде будет отображаться текущая страница, т.е. на какой язык будут переведены
оригинальные сообщения. Этот параметр определяется в конфигурации приложения:

```php
return [
    'id' => 'applicationID',
    'basePath' => dirname(__DIR__),
    // ...
    'language' => 'ru-RU', // <- здесь!
    // ...
]
```

> Tip: значение по умолчанию для [[yii\base\Application::$sourceLanguage|исходного языка]] — английский.

Вы можете установить значение текущего языка в самом приложении в соответствии с языком, который выбрал пользователь.
Это необходимо сделать до того, как будет сгенерирован какой-либо вывод, чтобы не возникло проблем с его 
корректностью. Используйте простое переопределение свойства на нужное значение:

```php
\Yii::$app->language = 'ru-RU';
```

Формат для установки языка/локали: `ll-CC`, где `ll` — это двух или трёхбуквенный код языка в нижнем регистре в 
соответствии со стандартом [ISO-639](https://www.loc.gov/standards/iso639-2/), а `CC` — это код страны в соответствии
со стандартом [ISO-3166](https://ru.wikipedia.org/wiki/ISO_3166-1#Список_кодов_по_ISO_3166).

> Note: больше информации о синтаксисе и концепции локалей можно получить в [документации проекта ICU](https://unicode-org.github.io/icu/userguide/locale/#the-locale-concept).

Перевод сообщений <span id="message-translation"></span>
--------------------------------------------------------

Перевод используется для локализации сообщений, которые будут выведены в приложении в соответствии с языком,
который выбрал пользователь.

По сути, Yii просто находит в файле с сообщениями на выбранном языке строку, соответствующую сообщению на исходном 
языке приложения. Для перевода сообщений необходимо в самом приложении заключать их в метод [[Yii::t()]]. Первый 
аргумент метода — это категория, которая позволяет группировать сообщения по определённому признаку, а второй — само 
сообщение.

```php
echo \Yii::t('app', 'This is a string to translate!');
```

Yii попытается загрузить файл перевода сообщений, соответствующий  текущему [[yii\base\Application::$language|языку приложения]]
из одного из источников, определённых в `i18n` [компонентах приложения](concept-components.md). Сообщения — это набор
файлов или база данных, которая содержит переведённые строки. Следующая конфигурация определяет, что сообщения 
должны браться из PHP-файлов:

```php
'components' => [
    // ...
    'i18n' => [
        'translations' => [
            'app*' => [
                'class' => 'yii\i18n\PhpMessageSource',
                //'basePath' => '@app/messages',
                //'sourceLanguage' => 'en-US',
                'fileMap' => [
                    'app'       => 'app.php',
                    'app/error' => 'error.php',
                ],
            ],
        ],
    ],
],
```

В приведённой конфигурации, `app*` — это шаблон, который определяет, какие категории обрабатываются источником. В нашем 
случае мы обрабатываем все, что начинается с `app`. Файлы с сообщениями находятся в `@app/messages` (папке `messages`
в вашем приложении). Массив [[yii\i18n\PhpMessageSource::fileMap|fileMap]] определяет, какой файл будет подключаться для
определённой категории. Если вы не хотите конфигурировать `fileMap`, можно положиться на соглашение, что название 
категории является именем файла. Например, категория `app/error` относится к файлу `app/error.php` в рамках [[yii\i18n\PhpMessageSource::basePath|basePath]].

Переводя сообщение `\Yii::t('app', 'This is a string to translate!')` при установленном языке приложения `ru-RU`, Yii
сначала будет искать файл `@app/messages/ru-RU/app.php`, чтобы получить список доступных переводов. Если такого файла нет в папке
`ru-RU`, Yii также попробует поискать в папке `ru` перед тем, как примет решение, что попытка перевода не удалась.

Кроме хранения в PHP-файлах (используя [[yii\i18n\PhpMessageSource|PhpMessageSource]]), Yii предоставляет ещё два
класса:

- [[yii\i18n\GettextMessageSource]], использующий GNU Gettext для MO или PO файлов.
- [[yii\i18n\DbMessageSource]], использующий базу данных.


### Именованные указатели

Вы можете добавлять параметры в строку для перевода, которые в выводе будут заменены соответствующими значениями, 
заключая параметр в фигурные скобки:

```php
$username = 'Alexander';
echo \Yii::t('app', 'Hello, {username}!', [
    'username' => $username,
]);
```

Обратите внимание, что в операции присваивания фигурные скобки не используются.

### Позиционные указатели 

```php
$sum = 42;
echo \Yii::t('app', 'Balance: {0}', $sum);
```

> Tip: старайтесь сохранять читаемость сообщений и избегать избыточного использования позиционных
параметров. Помните, что переводчик, скорее всего, будет располагать только файлом со строками и для него
должно быть очевидно, на что будет заменён тот или иной указатель.

### Указатели с расширенным форматированием

Чтобы использовать расширенные возможности, вам необходимо установить и включить [PHP-расширение intl](https://www.php.net/manual/ru/intro.intl.php).
После этого вам станет доступен расширенный синтаксис указателей, а также сокращённая запись `{placeholderName, argumentType}`,
эквивалентная форме `{placeholderName, argumentType, argumentStyle}`, позволяющая определять стиль форматирования.

Полная документация доступна на [сайте ICU](https://unicode-org.github.io/icu-docs/apidoc/released/icu4c/classMessageFormat.html), но далее в 
документации будут приведены примеры использования расширенных возможностей интернационализации.

#### Числа

```php
$sum = 42;
echo \Yii::t('app', 'Balance: {0, number}', $sum);
```

Вы можете использовать один из встроенных форматов (`integer`, `currency`, `percent`):

```php
$sum = 42;
echo \Yii::t('app', 'Balance: {0, number, currency}', $sum);
```

Или определить свой формат:

```php
$sum = 42;
echo \Yii::t('app', 'Balance: {0, number, ,000,000000}', $sum);
```

[Описание форматирования](https://unicode-org.github.io/icu-docs/apidoc/released/icu4c/classDecimalFormat.html).

#### Даты

```php
echo \Yii::t('app', 'Today is {0, date}', time());
```

Встроенные форматы — это `short`, `medium`, `long` и `full`:

```php
echo \Yii::t('app', 'Today is {0, date, short}', time());
```

Используя свой формат:

```php
echo \Yii::t('app', 'Today is {0, date,yyyy-MM-dd}', time());
```

[Описание форматирования](https://unicode-org.github.io/icu-docs/apidoc/released/icu4c/classicu_1_1SimpleDateFormat.html#details).

#### Время

```php
echo \Yii::t('app', 'It is {0, time}', time());
```

Встроенные форматы — это `short`, `medium`, `long` и `full`:

```php
echo \Yii::t('app', 'It is {0, time, short}', time());
```

Используя свой формат:

```php
echo \Yii::t('app', 'It is {0, date,HH:mm}', time());
```

[Описание форматирования](https://unicode-org.github.io/icu-docs/apidoc/released/icu4c/classicu_1_1SimpleDateFormat.html#details).


#### Числа прописью

```php
echo \Yii::t('app', 'Число {n,number} прописью: {n, spellout}', ['n' => 42]);
```

#### Порядковые числительные

```php
echo \Yii::t('app', 'Вы — {n, ordinal} посетитель!', ['n' => 42]);
```

Выведет сообщение "Вы — 42-й посетитель!".

#### Продолжительность

```php
echo \Yii::t('app', 'Вы находитесь здесь уже {n, duration}', ['n' => 47]);
```

Выведет сообщение "Вы находитесь здесь уже 47 сек.".

#### Множественное число

В каждом языке используется свой способ склонения порядковых числительных. Некоторые правила весьма сложны,
так что очень удобно, что использование функционала i18n не требует определения правил склонения. Требуется
только указать формы склоняемого слова в различных ситуациях:

```php
echo \Yii::t(
    'app',
    'На диване {n, plural, =0{нет кошек} =1{лежит одна кошка} one{лежит # кошка} few{лежит # кошки} many{лежит # кошек} other{лежит # кошки}}!', 
    ['n' => 0]
);
```

Выведет сообщение "На диване нет кошек!".

В данном правиле 

* `=0` означает ноль; 
* `=1` соответствует ровно `1`; 
* `one` — `21`, `31`, `41` и так далее; 
* `few` — от `2` до `4`, от `22` до `24` и так далее; 
* `many` — `0`, от `5` до `20`, от `25` до `30` и так далее; 
* `other` — для всех прочих чисел (например, дробных). 
* Решётка `#` заменяется на значение аргумента `n`. 

Для некоторых языков правила могут быть более простыми. Например, для английского будет достаточно указать:

```php
echo \Yii::t('app', 'There {n, plural, =0{are no cats} =1{is one cat} other{are # cats}}!', ['n' => 0]);
```

Следует помнить, что если вы используете указатель дважды, и в первый раз он используется как `plural`,
второй раз он должен быть использован как `number`, иначе вы получите ошибку "Inconsistent types declared 
for an argument: U_ARGUMENT_TYPE_MISMATCH":

```
В корзине: {count, number} {count, plural, one{товар} few{товара} other{товаров}}.
```

Подробная документация о формах склонений для различных языков доступна на сайте 
[unicode.org](https://cldr.unicode.org/index/cldr-spec/plural-rules).

#### Вариации

Вы можете указывать критерии форматирования сообщений в зависимости от ключевых слов. Приведённый пример
демонстрирует возможность подстановки корректного рода в зависимости от параметра:

```php
echo \Yii::t('app', '{name} — {gender} и {gender, select, женщина{ей} мужчина{ему} other{ему}} нравится Yii!', [
    'name'   => 'Василий',
    'gender' => 'мужчина',
]);
```

Выведет сообщение "Василий — мужчина и ему нравится Yii!".

В приведённом выражении, `мужчина` и `женщина` — это возможные варианты пола. На всякий случай, `other`
обработает случай, если значение не совпадает с первыми двумя вариантами. Строки в скобках являются вторичными
выражениями и могут быть просто строкой или строкой, содержащей дополнительные указатели.

### Определение перевода по умолчанию

Вы можете определить переводы, которые будут использованы как переводы по умолчанию для категорий, которые
не попадают в другие переводы. Этот перевод должен быть помечен звёздочкой `*` и указан в конфигурации
приложения как:

```php
// конфигурация i18n компонента

'i18n' => [
    'translations' => [
        '*' => [
            'class' => 'yii\i18n\PhpMessageSource'
        ],
    ],
],
```

Теперь можно использовать категории без необходимости конфигурировать каждую из них, что похоже на 
способ, которым была реализована поддержка интернационализации в Yii 1.1. Сообщения для категории будут
загружаться из файла с переводом по умолчанию из `basePath`, т.е. `@app/messages`:

```php
echo Yii::t('not_specified_category', 'message from unspecified category');
```

Сообщение будет загружено из файла `@app/messages/<LanguageCode>/not_specified_category.php`

### Перевод сообщений модулей

Если вы хотите перевести сообщения в модуле и при этом не сгружать их все в один файл, можете прибегнуть
к следующему приёму:

```php
<?php

namespace app\modules\users;

use Yii;

class Module extends \yii\base\Module
{
    public $controllerNamespace = 'app\modules\users\controllers';

    public function init()
    {
        parent::init();
        $this->registerTranslations();
    }

    public function registerTranslations()
    {
        Yii::$app->i18n->translations['modules/users/*'] = [
            'class'          => 'yii\i18n\PhpMessageSource',
            'sourceLanguage' => 'en-US',
            'basePath'       => '@app/modules/users/messages',
            'fileMap'        => [
                'modules/users/validation' => 'validation.php',
                'modules/users/form'       => 'form.php',
                ...
            ],
        ];
    }

    public static function t($category, $message, $params = [], $language = null)
    {
        return Yii::t('modules/users/' . $category, $message, $params, $language);
    }

}
```

В приведённом примере мы использовали маску для поиска совпадений, и последующую фильтрацию по
категориям для искомого файла. Вместо использования `fileMap`, вы можете прибегнуть к соглашению,
что имя категории совпадает с именем файла и писать `Module::t('validation', 'your custom validation message')` или
`Module::t('form', 'some form label')` напрямую.

### Перевод сообщений виджетов

Для виджетов применимо такое же правило, как и для модулей:

```php
<?php

namespace app\widgets\menu;

use yii\base\Widget;
use Yii;

class Menu extends Widget
{

    public function init()
    {
        parent::init();
        $this->registerTranslations();
    }

    public function registerTranslations()
    {
        $i18n = Yii::$app->i18n;
        $i18n->translations['widgets/menu/*'] = [
            'class'          => 'yii\i18n\PhpMessageSource',
            'sourceLanguage' => 'en-US',
            'basePath'       => '@app/widgets/menu/messages',
            'fileMap'        => [
                'widgets/menu/messages' => 'messages.php',
            ],
        ];
    }

    public function run()
    {
        echo $this->render('index');
    }

    public static function t($category, $message, $params = [], $language = null)
    {
        return Yii::t('widgets/menu/' . $category, $message, $params, $language);
    }

}
```
Вместо использования `fileMap`, вы можете прибегнуть к соглашению, что имя категории совпадает с 
именем файла и писать `Menu::t('messages', 'new messages {messages}', ['{messages}' => 10])` напрямую.

> Note: для виджетов вы можете использовать i18n представления. На них распространяются
> те же правила, что и на контроллеры.


### Перевод сообщений фреймворка

Yii поставляется с набором сообщений по умолчанию для ошибок валидации и некоторых других строк. Эти 
сообщения принадлежат категории `yii`. Если возникает необходимость изменить сообщения по умолчанию, 
переопределите `i18n` [компонент приложения](concept-components.md):

```php
'i18n' => [
    'translations' => [
        'yii' => [
            'class' => 'yii\i18n\PhpMessageSource',
            'sourceLanguage' => 'en-US',
            'basePath' => '@app/messages'
        ],
    ],
],
```

После этого разместите изменённые строки в файле `@app/messages/<language>/yii.php`.

### Обработка недостающих переводов

Если в источнике перевода отсутствует необходимое сообщение, Yii отобразит исходное содержимое сообщения.
Данное поведение тем оправданнее, чем вы более стремитесь писать в исходном коде понятный текст сообщений.
Тем не  менее, иногда этого недостаточно, и может потребоваться произвольная обработка возникшей ситуации, 
когда источник не содержит искомой строки. Для этого следует использовать обработку события 
[[yii\i18n\MessageSource::EVENT_MISSING_TRANSLATION|missingTranslation]] компонента [[yii\i18n\MessageSource]].

Например, чтобы отметить все непереведённые строки, чтобы их было легче находить на странице, необходимо 
создать обработчик события. Изменим конфигурацию приложения:

```php
'components' => [
    // ...
    'i18n' => [
        'translations' => [
            'app*' => [
                'class' => 'yii\i18n\PhpMessageSource',
                'fileMap' => [
                    'app' => 'app.php',
                    'app/error' => 'error.php',
                ],
                'on missingTranslation' => ['app\components\TranslationEventHandler', 'handleMissingTranslation']
            ],
        ],
    ],
],
```

Создадим обработчик события:

```php
<?php

namespace app\components;

use yii\i18n\MissingTranslationEvent;

class TranslationEventHandler
{
    public static function handleMissingTranslation(MissingTranslationEvent $event) {
        $event->translatedMessage = "@MISSING: {$event->category}.{$event->message} FOR LANGUAGE {$event->language} @";
    }
}
```

Если [[yii\i18n\MissingTranslationEvent::translatedMessage]] установлен как обработчик события, на странице будет
выведен соответствующий результат перевода.

> Warning: каждый источник обрабатывает недостающие переводы самостоятельно. Если вы используете несколько разных
> источников сообщений и хотите обрабатывать недостающие переводы одинаково для всех, назначьте соответствующий
> обработчик события для каждого источника.


Использование команды `message` <span id="message-command"></span>
--------------------------------------------------------------

Переводы могут храниться в [[yii\i18n\PhpMessageSource|PHP-файлах]], [[yii\i18n\GettextMessageSource|файлах .po]] или в [[yii\i18n\DbMessageSource|базе данных]]. См. соответствующие классы для дополнительных опций.

Прежде всего, вам необходимо создать конфигурационный файл. Решите, где вы хотите хранить его и затем выполните команду

```bash
./yii message/config-template path/to/config.php
```

Откройте созданный файл и настройте параметры в соответствии со своими потребностями. Уделите особое внимание следующим параметрам:

* `languages`: массив, содержащий языки, на которые ваше приложение должно быть переведено;
* `messagePath`: путь для хранения файлов сообщений, который должен соответствовать параметру `basePath`, указанному в конфигурации компонента `i18n`.

Вы также можете использовать команду `./yii message/config`, чтобы динамически сгенерировать конфигурационный файл с указанными опциями с помощью командной строки.
Например, вы можете установить параметры `languages` и `messagePath` следующим образом:

```bash
./yii message/config --languages=de,ja --messagePath=messages path/to/config.php
```

Чтобы получить список доступных параметров, выполните следующую команду:

```bash
./yii help message/config
```

Как только вы закончили с файлом конфигурации, вы можете, наконец, извлечь свои сообщения с помощью команды:

```bash
./yii message path/to/config.php
```

Также вы можете использовать параметры, чтобы динамически менять настройки извлечения.

В результате вы найдете свои файлы (если вы выбрали перевод с помощью файлов) в своей директории `messagePath`.


Представления <span id="view-translation"></span>
-------------------------------------------------

Вместо того, чтобы переводить сообщения так, как указано в предыдущем разделе, вы можете использовать `i18n` в ваших
представлениях, чтобы обеспечить поддержку нескольких языков. Например, если существует представление `views/site/index.php`
и для перевода его на русский язык необходимо отдельное представление, создайте папку `ru-RU` в папке с представлением
текущего контроллера или виджета и создайте файл для русского языка: `views/site/ru-RU/index.php`. Yii загрузит файл
для текущего языка, если он существует, или использует исходный `views/site/index.php`, если не сможет найти локализацию.

> Note: если язык был определён, как `en-US` и соответствующих представлений не было найдено, Yii попробует
> найти представления в папке `en` перед тем, как использовать исходные.


Форматирование чисел и дат <span id="date-number"></span>
---------------------------------------------------------

См. описание [форматирования дат](output-formatting.md).


Настройка PHP-окружения <span id="setup-environment"></span>
------------------------------------------------------------

Для работы с большей частью функций интернационализации Yii использует [PHP-расширение intl](https://www.php.net/manual/ru/book.intl.php).
Например, это расширение используют классы, отвечающие за форматирование чисел и дат [[yii\i18n\Formatter]] и за форматирование
строк [[yii\i18n\MessageFormatter]]. Оба класса поддерживают базовый функционал даже в том случае, если расширение `intl` не
установлено. Однако этот запасной вариант более или менее будет работать только для сайтов на английском языке, хотя даже для
них большая часть широких возможностей расширения `intl` не будет доступна, поэтому его установка настоятельно рекомендуется.

[PHP-расширение intl](https://www.php.net/manual/ru/book.intl.php) основано на [библиотеке ICU](https://icu.unicode.org/), которая
описывает правила форматирования для различных локалей. Поэтому следует помнить, что форматирование чисел и дат вместе с
синтаксисом форматирования может отличаться в зависимости от версии библиотеки ICU, которая была скомпилирована в вашем
дистрибутиве PHP.

Чтобы сайт работал одинаково во всех окружениях, рекомендуется устанавливать одинаковую версию расширения intl, при этом 
удостоверяясь, что везде используется одинаковая версия библиотеки ICU.

Чтобы узнать, какая версия ICU используется текущим PHP интерпретатором, используйте следующий скрипт:

```php
<?php
echo "PHP: " . PHP_VERSION . "\n";
echo "ICU: " . INTL_ICU_VERSION . "\n";
```

Чтобы иметь доступ ко всем возможностям, описанным в документации, мы рекомендуем использовать ICU версии 49 или новее.
В более ранних версиях отсутствует указатель `#` в правилах склонений. На сайте <https://icu.unicode.org/download>
вы можете ознакомиться со списком доступных версий ICU. Обратите внимание, что схема нумерации версий изменилась после
версии 4.8 и последовательность версий выглядит так: ICU 4.8, ICU 49, ICU 50, ICU 51 и так далее.

Известные проблемы <span id="known-problems"></span>
----------------------------------------------------

 - В ICU версии 52.1 было испорчено форматирование множественных чисел (`plural`) в русском языке. Проблема решается обновлением ICU до версии 53.1 или старше.
