<?php

namespace Winter\Storm\Database\Relations\Concerns;

use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;

trait BelongsOrMorphsToMany
{
        /**
     * @var boolean This relation object is a 'count' helper.
     */
    public $countMode = false;

    /**
     * @var boolean When a join is not used, don't select aliased columns.
     */
    public $orphanMode = false;

    /**
     * Get the select columns for the relation query.
     *
     * @param  array  $columns
     * @return array|string
     */
    protected function shouldSelect(array $columns = ['*'])
    {
        if ($this->countMode) {
            return $this->table.'.'.$this->foreignPivotKey.' as pivot_'.$this->foreignPivotKey;
        }

        if ($columns == ['*']) {
            $columns = [$this->related->getTable().'.*'];
        }

        if ($this->orphanMode) {
            return $columns;
        }

        return array_merge($columns, $this->aliasedPivotColumns());
    }

    /**
     * Save the supplied related model with deferred binding support.
     */
    public function save(Model $model, array $pivotData = [], $sessionKey = null)
    {
        $model->save();
        $this->add($model, $sessionKey, $pivotData);
        return $model;
    }

    /**
     * Override sync() method of BelongToMany relation in order to flush the query cache.
     * @param array $ids
     * @param bool $detaching
     * @return array
     */
    public function sync($ids, $detaching = true)
    {
        $changed = parent::sync($ids, $detaching);

        $this->flushDuplicateCache();

        return $changed;
    }

    /**
     * Create a new instance of this related model with deferred binding support.
     */
    public function create(array $attributes = [], array $pivotData = [], $sessionKey = null)
    {
        $model = $this->related->create($attributes);

        $this->add($model, $sessionKey, $pivotData);

        return $model;
    }

    /**
     * Override attach() method of BelongToMany relation.
     * This is necessary in order to fire 'model.relation.beforeAttach', 'model.relation.afterAttach' events
     * @param mixed $id
     * @param array $attributes
     * @param bool  $touch
     */
    public function attach($id, array $attributes = [], $touch = true)
    {
        $insertData = $this->formatAttachRecords($this->parseIds($id), $attributes);
        $attachedIdList = array_pluck($insertData, $this->relatedPivotKey);

        $eventArgs = [$this->relationName];

        if ($this->using) {
            $eventArgs = [...$eventArgs, $id, $attributes];
        } else {
            $eventArgs = [...$eventArgs, $attachedIdList, $insertData];
        }

        /**
         * @event model.relation.beforeAttach
         * Called before creating a new relation between models (only for BelongsToMany relation)
         *
         * Example usage:
         *
         *     $model->bindEvent('model.relation.beforeAttach', function (string $relationName, array $attachedIdList, array $insertData) use (\Winter\Storm\Database\Model $model) {
         *         if (!$model->isRelationValid($attachedIdList)) {
         *             throw new \Exception("Invalid relation!");
         *             return false;
         *         }
         *     });
         *
         * >**NOTE:** If a custom pivotModel is being used the parameters will actually be `string $relationName, mixed $id, array $attributes`
         *
         */
        if ($this->parent->fireEvent('model.relation.beforeAttach', $eventArgs, true) === false) {
            return;
        }

        // Here we will insert the attachment records into the pivot table. Once we have
        // inserted the records, we will touch the relationships if necessary and the
        // function will return. We can parse the IDs before inserting the records.
        if ($this->using) {
            $this->attachUsingCustomClass($id, $attributes);
        } else {
            $this->newPivotStatement()->insert($insertData);
        }

        if ($touch) {
            $this->touchIfTouching();
        }

        /**
         * @event model.relation.afterAttach
         * Called after creating a new relation between models (only for BelongsToMany relation)
         *
         * Example usage:
         *
         *     $model->bindEvent('model.relation.afterAttach', function (string $relationName, array $attachedIdList, array $insertData) use (\Winter\Storm\Database\Model $model) {
         *         traceLog("New relation {$relationName} was created", $attachedIdList);
         *     });
         *
         * >**NOTE:** If a custom pivotModel is being used the parameters will actually be `string $relationName, mixed $id, array $attributes`
         *
         */
        $this->parent->fireEvent('model.relation.afterAttach', $eventArgs);
    }

    /**
     * Override detach() method of BelongToMany relation.
     * This is necessary in order to fire 'model.relation.beforeDetach', 'model.relation.afterDetach' events
     * @param Collection|Model|array|null $ids
     * @param bool $touch
     * @return int|void
     */
    public function detach($ids = null, $touch = true)
    {
        $attachedIdList = $this->parseIds($ids);
        if (empty($attachedIdList)) {
            $attachedIdList = $this->newPivotQuery()->lists($this->relatedPivotKey);
        }

        /**
         * @event model.relation.beforeDetach
         * Called before removing a relation between models (only for BelongsToMany relation)
         *
         * Example usage:
         *
         *     $model->bindEvent('model.relation.beforeDetach', function (string $relationName, array $attachedIdList) use (\Winter\Storm\Database\Model $model) {
         *         if (!$model->isRelationValid($attachedIdList)) {
         *             throw new \Exception("Invalid relation!");
         *             return false;
         *         }
         *     });
         *
         */
        if ($this->parent->fireEvent('model.relation.beforeDetach', [$this->relationName, $attachedIdList], true) === false) {
            return;
        }

        /**
         * @see Illuminate\Database\Eloquent\Relations\Concerns\InteractsWithPivotTable
         */
        parent::detach($attachedIdList, $touch);

        /**
         * @event model.relation.afterDetach
         * Called after removing a relation between models (only for BelongsToMany relation)
         *
         * Example usage:
         *
         *     $model->bindEvent('model.relation.afterDetach', function (string $relationName, array $attachedIdList) use (\Winter\Storm\Database\Model $model) {
         *         traceLog("Relation {$relationName} was removed", $attachedIdList);
         *     });
         *
         */
        $this->parent->fireEvent('model.relation.afterDetach', [$this->relationName, $attachedIdList]);
    }

    /**
     * Adds a model to this relationship type.
     */
    public function add(Model $model, $sessionKey = null, $pivotData = [])
    {
        if (is_array($sessionKey)) {
            $pivotData = $sessionKey;
            $sessionKey = null;
        }

        if ($sessionKey === null || $sessionKey === false) {
            $this->attach($model, $pivotData);
            $this->parent->reloadRelations($this->relationName);
        } else {
            $this->parent->bindDeferred($this->relationName, $model, $sessionKey, $pivotData);
        }
    }

    /**
     * Removes a model from this relationship type.
     */
    public function remove(Model $model, $sessionKey = null)
    {
        if ($sessionKey === null) {
            $this->detach($model);
            $this->parent->reloadRelations($this->relationName);
        }
        else {
            $this->parent->unbindDeferred($this->relationName, $model, $sessionKey);
        }
    }

    /**
     * Get a paginator for the "select" statement. Complies with Winter Storm.
     *
     * @param  int    $perPage
     * @param  int    $currentPage
     * @param  array  $columns
     * @param  string  $pageName
     * @return \Illuminate\Contracts\Pagination\LengthAwarePaginator
     */
    public function paginate($perPage = 15, $currentPage = null, $columns = ['*'], $pageName = 'page')
    {
        $this->query->addSelect($this->shouldSelect($columns));

        $paginator = $this->query->paginate($perPage, $currentPage, $columns, $pageName);

        $this->hydratePivotRelation($paginator->items());

        return $paginator;
    }

    /**
     * Paginate the given query into a simple paginator.
     *
     * @param  int|null  $perPage
     * @param  int|null  $currentPage
     * @param  array  $columns
     * @param  string  $pageName
     * @return \Illuminate\Contracts\Pagination\Paginator
     */
    public function simplePaginate($perPage = null, $currentPage = null, $columns = ['*'], $pageName = 'page')
    {
        $this->query->addSelect($this->shouldSelect($columns));

        $paginator = $this->query->simplePaginate($perPage, $currentPage, $columns, $pageName);

        $this->hydratePivotRelation($paginator->items());

        return $paginator;
    }

    /**
     * Create a new pivot model instance.
     *
     * @param  array  $attributes
     * @param  bool   $exists
     * @return \Illuminate\Database\Eloquent\Relations\Pivot
     */
    public function newPivot(array $attributes = [], $exists = false)
    {
        /*
         * Winter looks to the relationship parent
         */
        $pivot = $this->parent->newRelationPivot($this->relationName, $this->parent, $attributes, $this->table, $exists);

        /*
         * Laravel looks to the related model
         */
        if (empty($pivot)) {
            $pivot = $this->related->newPivot($this->parent, $attributes, $this->table, $exists);
        }

        return $pivot->setPivotKeys($this->foreignPivotKey, $this->relatedPivotKey);
    }

    /**
     * Get all of the IDs for the related models, with deferred binding support
     *
     * @param string $sessionKey
     * @return \Illuminate\Support\Collection
     */
    public function allRelatedIds($sessionKey = null)
    {
        $related = $this->getRelated();

        $fullKey = $related->getQualifiedKeyName();

        $query = $sessionKey ? $this->withDeferred($sessionKey) : $this;

        return $query->getQuery()->select($fullKey)->pluck($related->getKeyName());
    }

    /**
     * Get the fully qualified foreign key for the relation.
     *
     * @return string
     */
    public function getForeignKey()
    {
        return $this->table.'.'.$this->foreignPivotKey;
    }

    /**
     * Get the fully qualified "other key" for the relation.
     *
     * @return string
     */
    public function getOtherKey()
    {
        return $this->table.'.'.$this->relatedPivotKey;
    }

    /**
     * @deprecated Use allRelatedIds instead. Remove if year >= 2018.
     */
    public function getRelatedIds($sessionKey = null)
    {
        traceLog('Method BelongsToMany::getRelatedIds has been deprecated, use BelongsToMany::allRelatedIds instead.');
        return $this->allRelatedIds($sessionKey)->all();
    }
}
