<?php declare(strict_types=1);

namespace Shopware\Elasticsearch;

use OpenSearchDSL\BuilderInterface;
use OpenSearchDSL\Query\Compound\BoolQuery;
use OpenSearchDSL\Query\Compound\DisMaxQuery;
use OpenSearchDSL\Query\FullText\MatchPhrasePrefixQuery;
use OpenSearchDSL\Query\FullText\MatchQuery;
use OpenSearchDSL\Query\Joining\NestedQuery;
use OpenSearchDSL\Query\TermLevel\PrefixQuery;
use OpenSearchDSL\Query\TermLevel\TermQuery;
use OpenSearchDSL\Query\TermLevel\TermsQuery;
use Shopware\Core\Framework\Adapter\Storage\AbstractKeyValueStorage;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\Dbal\EntityDefinitionQueryHelper;
use Shopware\Core\Framework\DataAbstractionLayer\DefinitionInstanceRegistry;
use Shopware\Core\Framework\DataAbstractionLayer\Field\Field;
use Shopware\Core\Framework\DataAbstractionLayer\Field\FloatField;
use Shopware\Core\Framework\DataAbstractionLayer\Field\IntField;
use Shopware\Core\Framework\DataAbstractionLayer\Field\ListField;
use Shopware\Core\Framework\DataAbstractionLayer\Field\LongTextField;
use Shopware\Core\Framework\DataAbstractionLayer\Field\PriceField;
use Shopware\Core\Framework\DataAbstractionLayer\Field\StringField;
use Shopware\Core\Framework\DataAbstractionLayer\Field\TranslatedField;
use Shopware\Core\Framework\Feature;
use Shopware\Core\Framework\Log\Package;
use Shopware\Core\System\CustomField\CustomFieldService;
use Shopware\Elasticsearch\Product\ElasticsearchOptimizeSwitch;
use Shopware\Elasticsearch\Product\SearchFieldConfig;

/**
 * @phpstan-type SearchConfig array{and_logic: string, field: string, tokenize: int, ranking: float|int}
 *
 * @final
 */
#[Package('inventory')]
class TokenQueryBuilder
{
    /**
     * @internal
     */
    public function __construct(
        private readonly DefinitionInstanceRegistry $definitionRegistry,
        private readonly CustomFieldService $customFieldService,
        private readonly AbstractKeyValueStorage $storage,
        private readonly int $minGram = 4,
        private readonly bool $useLanguageAnalyzer = true
    ) {
    }

    /**
     * @param SearchFieldConfig[] $configs
     */
    public function build(string $entity, string $token, array $configs, Context $context): ?BuilderInterface
    {
        $token = mb_strtolower(trim($token));
        $languageIdChain = $context->getLanguageIdChain();
        $explainMode = $context->hasState(Context::ELASTICSEARCH_EXPLAIN_MODE);

        $tokenQueries = [];

        $definition = $this->definitionRegistry->getByEntityName($entity);

        foreach ($configs as $config) {
            $field = EntityDefinitionQueryHelper::getField($config->getField(), $definition, $definition->getEntityName(), false);
            $fieldDefinition = EntityDefinitionQueryHelper::getAssociatedDefinition($definition, $config->getField());
            $real = $field instanceof TranslatedField ? EntityDefinitionQueryHelper::getTranslatedField($fieldDefinition, $field) : $field;

            if (str_contains($config->getField(), 'customFields')) {
                $real = $this->customFieldService->getCustomField(str_replace('customFields.', '', $config->getField()));
            }

            if (!$real) {
                continue;
            }

            $root = EntityDefinitionQueryHelper::getRoot($config->getField(), $definition);

            $fieldQuery = $field instanceof TranslatedField ?
                // If the field is a TranslatedField, we need to build a translated query
                // translated query will use the languageIdChain to find the correct translation with fallback
                // and if the field is prefilled fallback, we can use the current languageId as every languageId is filled with the fallback when indexing
                $this->translatedQuery($real, $token, $config, $this->isSortableTranslatedField($field) ? [$context->getLanguageId()] : $languageIdChain) :
                $this->matchQuery($real, $token, $config);

            if (!$fieldQuery) {
                continue;
            }

            if ($root !== null) {
                $fieldQuery = new NestedQuery($root, $fieldQuery);
            }

            if ($explainMode) {
                $fieldQuery = $this->explainQuery($token, $fieldQuery, $config);
            }

            $tokenQueries[] = $fieldQuery;
        }

        if (empty($tokenQueries)) {
            return null;
        }

        if (\count($tokenQueries) === 1) {
            return $tokenQueries[0];
        }

        return new BoolQuery([BoolQuery::SHOULD => $tokenQueries]);
    }

    private function matchQuery(Field $field, string $token, SearchFieldConfig $config): ?BuilderInterface
    {
        if ($field instanceof StringField || $field instanceof LongTextField || $field instanceof ListField) {
            $queries = [];

            $searchField = $config->getField() . '.search';
            $operator = $config->isAndLogic() ? 'and' : 'or';

            $tokens = preg_split('/\s+/u', $token, -1, \PREG_SPLIT_NO_EMPTY) ?: [$token];
            $tokenCount = \count($tokens);

            if ($tokenCount > 1) {
                $token = implode(' ', $tokens);
            }

            // apply exact match
            $queries[] = $tokenCount === 1
                ? new TermQuery($config->getField(), $token, ['boost' => 1])
                : new TermsQuery($config->getField(), $tokens, ['boost' => 1]);

            $lastWord = array_last($tokens);
            $maxExpansions = $this->getMaxExpansions($lastWord);

            // apply fuzzy search
            $matchQueryParams = [
                'boost' => 0.8,
                'fuzziness' => $config->getFuzziness($token),
                'operator' => $operator,
                'fuzzy_transpositions' => true, // treats "ab" and "ba" as a single edit
                'max_expansions' => $maxExpansions, // limit the number of variations
                'prefix_length' => 1, // reduce noise
            ];

            if (!$this->useLanguageAnalyzer) {
                $matchQueryParams['analyzer'] = 'sw_whitespace_analyzer';
            }

            $queries[] = new MatchQuery($searchField, $token, $matchQueryParams);

            // apply match phrase prefix for compound tokens
            if ($config->usePrefixMatch()) {
                // apply prefix search on a single token or match phrase prefix on multiple tokens
                if ($tokenCount > 1) {
                    $matchPhrasePrefixParams = [
                        'boost' => 0.6,
                        'slop' => 3,
                        'max_expansions' => $maxExpansions,
                    ];

                    if (!$this->useLanguageAnalyzer) {
                        $matchPhrasePrefixParams['analyzer'] = 'sw_whitespace_analyzer';
                    }

                    $queries[] = new MatchPhrasePrefixQuery($searchField, $token, $matchPhrasePrefixParams);
                } else {
                    $queries[] = new PrefixQuery($config->getField(), $token, [
                        'boost' => 0.4,
                    ]);
                }
            }

            $tokenLength = mb_strlen($token);

            if ($config->tokenize() && $tokenCount === 1 && $tokenLength >= $this->minGram) {
                $queries[] = new MatchQuery($config->getField() . '.ngram', $token, [
                    'boost' => 0.4,
                ]);
            }

            $dismax = new DisMaxQuery();

            foreach ($queries as $query) {
                $dismax->addQuery($query);
            }

            $dismax->addParameter('boost', $config->getRanking());

            return $dismax;
        }

        if ($field instanceof IntField || $field instanceof FloatField || $field instanceof PriceField) {
            if (!\is_numeric($token)) {
                return null;
            }

            $token = $field instanceof IntField ? (int) $token : (float) $token;
        }

        return new TermQuery($config->getField(), $token, ['boost' => $config->getRanking()]);
    }

    /**
     * @param string[] $languageIdChain
     */
    private function translatedQuery(Field $field, string $token, SearchFieldConfig $config, array $languageIdChain): ?BuilderInterface
    {
        $languageQueries = [];

        $ranking = $config->getRanking();

        foreach ($languageIdChain as $languageId) {
            $searchField = $this->buildTranslatedFieldName($config, $languageId);

            $languageConfig = new SearchFieldConfig(
                $searchField,
                $ranking,
                $config->tokenize(),
                $config->isAndLogic(),
                $config->usePrefixMatch(),
            );

            $languageQuery = $this->matchQuery($field, $token, $languageConfig);

            $ranking *= 0.8; // for each language we go "deeper" in the translation, we reduce the ranking by 20%

            if (!$languageQuery) {
                continue;
            }

            $languageQueries[] = $languageQuery;
        }

        if (empty($languageQueries)) {
            return null;
        }

        if (\count($languageQueries) === 1) {
            return $languageQueries[0];
        }

        $dismax = new DisMaxQuery();

        foreach ($languageQueries as $languageQuery) {
            $dismax->addQuery($languageQuery);
        }

        return $dismax;
    }

    private function buildTranslatedFieldName(SearchFieldConfig $fieldConfig, string $languageId): string
    {
        if ($fieldConfig->isCustomField()) {
            $parts = explode('.', $fieldConfig->getField());

            return \sprintf('%s.%s.%s', $parts[0], $languageId, $parts[1]);
        }

        return \sprintf('%s.%s', $fieldConfig->getField(), $languageId);
    }

    private function explainQuery(string $token, BuilderInterface $fieldQuery, SearchFieldConfig $config): BuilderInterface
    {
        $explainPayload = json_encode([
            'field' => $config->getField(),
            'term' => $token,
            'ranking' => $config->getRanking(),
        ]);

        if (!method_exists($fieldQuery, 'addParameter')) {
            return $fieldQuery;
        }

        if ($fieldQuery instanceof NestedQuery) {
            $fieldQuery->addParameter('inner_hits', [
                '_source' => false,
                'explain' => true,
                'name' => $explainPayload,
            ]);
        }

        $fieldQuery->addParameter('_name', $explainPayload);

        return $fieldQuery;
    }

    private function isSortableTranslatedField(TranslatedField $field): bool
    {
        return $field->useForSorting() && (Feature::isActive('v6.8.0.0') || $this->storage->has(ElasticsearchOptimizeSwitch::FLAG));
    }

    /**
     * @see https://docs.opensearch.org/1.1/opensearch/query-dsl/full-text#options for max_expansions
     */
    private function getMaxExpansions(string $lastWord): int
    {
        $len = mb_strlen($lastWord);

        if ($len <= 3) {
            return 5;
        }

        if ($len <= 6) {
            return 10;
        }

        return 20;
    }
}
