<?php
namespace App;
/**
 * Core: Custom Field - Main Class
 * 
 * Core - Custom Field
 * 
 * @copyright 2019 SCHLIX Web Inc
 *
 * @license GPLv3
 *
 * @package core
 * @version 1.0
 * @author  SCHLIX Web Inc <info@schlix.com>
 * @link    http://www.schlix.com
 */
class Core_CustomField extends \SCHLIX\cmsApplication_CategorizedList {

    
    /**
     * Field types array
     * @var array 
     */
    protected  $field_type_categories = [];
    protected  $field_type_items = [];
    protected  $field_with_choices = [];
    
    public function __construct() {
 
        
        parent::__construct('Custom Field', 'gk_customfield_items', 'gk_customfield_tables');
        $this->field_type_categories =  [ 'string' => ___('String'), 'integer' => ___('Integer'),  'float' => ___('Floating Point'), 'datetime' => ___('Date and Time')] ;
        $this->field_type_items = [
            'string'   => ['VARCHAR', 'CHAR','MEDIUMTEXT','TEXT', 'LONGTEXT'],
            'integer'  => ['TINYINT', 'SMALLINT', 'INT', 'MEDIUMINT','BIGINT'],
            'float'    => ['DECIMAL','FLOAT','REAL','DOUBLE'],
            'datetime' => ['DATE','DATETIME', 'TIME', 'TIMESTAMP']
        ];
        $this->field_with_choices = ['dropdown','radiogroup','checkboxgroup','select','radio-group','checkbox-group'];
        $this->hook_priority = 9800;
        $this->schema_org_type_item = '';
        $this->has_versioning = true;
        $this->disable_frontend_runtime = true;
    }
            

    public function getFieldTypeCategories()
    {
        return $this->field_type_categories;
    }
    
    public function getFieldTypeItems($category)
    {
        return $this->field_type_items[$category];
    }
    
    /**
     * Returns the default category ID 
     * @return int
     */
    public function getDefaultCategoryID() {
        return 1;
    }  

    //_______________________________________________________________________________________________________________//
    /**
     * Returns an array containing on array of main page options. 
     * The values of the options will still be evaluated as a flat list array, 
     * however it is sectioned into array with the following keys:
     * header, value, type, and options.
     * - Label: section title (not used for any evaluation
     * - Type: checkboxgroup, dropdownlist, or none. If none, then it means there 
     *         are suboptions which contain another array of this
     * - Key: the key option. Please note that checkboxgroup doesn't have a key
     *        since the keys are in the options
     * - Options: an array with 2 keys: label and key
     * 
     * @return array
     */    
            
    /**
     * 
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $hash
     * @return int
     */
    public function getCategoryIDByTableName($table_name)
    {
        global $SystemDB;
        
        if ($SystemDB->tableExists($table_name))
        {

            //$sha = sha1($table_name);
            $table_param = ['table_name' => $table_name];
            $row = $SystemDB->getQueryResultSingleRow("SELECT * FROM {$this->table_categories} WHERE table_name = :table_name",$table_param);
            if (!$row)
            {
                $this->table_categories->quickInsert($table_param);
                $row = $SystemDB->getQueryResultSingleRow("SELECT * FROM {$this->table_categories} WHERE table_name = :table_name",$table_param);
                return $row['cid'];
            }
            return $row['cid'];
        }
        return null;
    }
    
    /**
     * Returns a list of custom fields
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $table_name
     * @return array
     */
    public function getCustomFields($table_name)
    {
        return self::getCustomFieldFromTable($table_name);
        
    }    
    
    /**
     * Returns a list of custom fields
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $table_name
     * @return array
     */
    public function getCustomFieldsByTableID($table_id)
    {
        global $SystemDB;
        
        return $SystemDB->getQueryResultArray("SELECT * FROM {$this->table_items}  WHERE category_id = :table_id", ['table_id' => $table_id]);
    }  
    
    /**
     * Returns a list of custom fields
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $table_name
     * @return array
     */
    public function getCustomFieldByName($field_name)
    {
        global $SystemDB;
        
        return $SystemDB->getQueryResultArray("SELECT * FROM {$this->table_items}  WHERE field_name = :field_name", ['field_name' => $field_name]);
    }    
    
    
    /**
     * 
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $hash
     * @return int
     */
    public function getAndCreateCategoryByTableHash($table_name)
    {
        global $SystemDB;
        
        $sha = sha1($table_name);
        $row = $SystemDB->getQueryResultSingleRow("SELECT * FROM {$this->table_categories} WHERE SHA1(table_name) = :sha",['sha' => $sha]);
        return $row['cid'];
    }
    
    private static function attrToString($attrs)
    {
        $arr_attrs = [];
        foreach ($attrs as $key => $value)
            if (strval($value) != '')
                $str_attrs.="{$key}=\"" . htmlspecialchars(strval($value)) . "\" ";
        return $str_attrs;
    }
    
    public static function getCustomFieldFromTable($table_name)
    {
        global $SystemDB;
        $cfs = $SystemDB->getQueryResultArray("SELECT * FROM gk_customfield_items INNER JOIN gk_customfield_tables ON  gk_customfield_items.category_id = gk_customfield_tables.cid WHERE table_name = :table_name", ['table_name' => $table_name]);
        return $cfs;
    }
    

    public static function getCustomFieldPOSTData($table_name)
    {
        $result = [];
        $custom_fields = self::getCustomFieldFromTable($table_name);
        if ($custom_fields)
        {
            foreach ($custom_fields as $cf)
            {
                $field_name = 'xcf_'.$cf['field_name'];
                if (array_key_exists($field_name, $_POST))
                {
                    $result[$field_name] = fpost_string($field_name);
                }
                
            }
        }
        return $result;
        
    }
    
    public static function displayAllCustomFieldFormInput($table_name, $is_admin = false)
    {
        $cfs = self::getCustomFieldFromTable($table_name);
        
        $s = '';
        foreach ($cfs as $cf)
        {
            $s.=self::drawCustomFieldFormInput($cf);
        }
        
        return $s;
        
    }
    
    function delete($mixed_items_to_delete)
    { // mixed item = categories + items
        //test case:
        //http://schlixcms/admin/index.php?page=html&ajax=1&action=ajax_delete&items=c3
        //http://schlixcms/admin/index.php?page=html&ajax=1&action=ajax_delete&items=c5,c6,c9,i4,i14
        global $SystemDB;
        
        
        $current_id = (int) substr($mixed_items_to_delete, 1);
        if ($current_id > 0)
        {
            $item = $this->getItemByID($current_id);
            if ($item)
            {
                $category = $this->getCategoryByID($item['category_id']);
                $table_name = $category['table_name'];
                $field_name = 'xcf_'.alpha_numeric_with_underscore($item['field_name']);
                if ($SystemDB->tableColumnExists($table_name, $field_name))
                {
                    parent::delete($mixed_items_to_delete);
                    $SystemDB->query("ALTER TABLE `{$table_name}` DROP `{$field_name}`");
                    $this->logInfo("A custom field has been deleted: [{$field_name}] from table [{$table_name}}");
                } else 
                {
                    $this->logError("Request to delete field: [{$field_name}] from table [{$table_name}} has failed because the field does not exist");
                }
            }
        }
    }
    
    public static function drawCustomFieldFormInput($cf, $is_admin = false)
    {
        $tag = '';
        $a = [];
        $a['data-field'] = $a['name'] = $a['id']   = 'xcf_'. $cf['field_name'];
        $a['label'] = $cf['field_label'];
        $select_opts = ['dropdown','select','radiogroup','radio-group','checkbox-group','checkboxgroup'];

        if (in_array($cf['field_box_ui'], $select_opts))
        {
            $options = [];
            if ($cf['field_choices'])
            {                
                $arr_choices = explode("\n", $cf['field_choices']);
                $options[] = ['value' => '', 'label' => ' -- '.___('Please select').' -- ' ];
                foreach ($arr_choices as $choice)
                {
                    list ($v, $l) = explode (';', $choice);
                    $l = trim($l);
                    $v = trim($v);
                    if (!empty ($l) && !empty($v))
                        $options[] = ['value' => $v, 'label' => $l];
                }
            }
        }        
        
        switch ($cf['field_box_ui'])
        {
            case 'checkbox':
                if ($cf['field_required'])
                    $a['required'] = 'required';                
                $str_attrs = self::attrToString($a);
                $tag =  "<x-ui:checkbox {$str_attrs} value=\"1\" />";
                break;
            case 'radio-group':
                 array_shift($options); //get rid of 1st blank option
            case 'select':
            
                if ($cf['field_required'])
                    $a['required'] = 'required';
                $str_attrs = self::attrToString($a);
                $tt = $cf['field_box_ui'];
                if ($tt == 'dropdown')
                    $tt = 'select';
                $tag = "<x-ui:{$tt} {$str_attrs}>";
                foreach ($options as $opt)
                {
                    $h_opt_value = ___h($opt['value']);
                    $h_opt_label = ___h($opt['label']);
                    $tag.= "<x-ui:option value=\"{$h_opt_value}\" label=\"{$h_opt_label}\" />";
                }
                $tag.= "</x-ui:{$tt}>";
                break;
                
            case 'textarea':
                if ($cf['field_required'])
                    $a['required'] = 'required';
                if ($cf['field_placeholder'])
                    $a['placeholder'] = $cf['field_placeholder'];
                $str_attrs = self::attrToString($a);
                $tag = "<x-ui:textarea {$str_attrs} />";                
                break;
            
            case 'wysiwyg':
                if ($cf['field_required'])
                    $a['required'] = 'required';                
                $str_attrs = self::attrToString($a);
                $tag = "<x-ui:wysiwyg {$str_attrs} />";                
                break;
            
            case 'numberbox':
                
                $a['type'] = 'number';
            
            default:
            case 'textbox':
                if ($cf['field_placeholder'])
                    $a['placeholder'] = $cf['field_placeholder'];                
                if ($cf['field_required'])
                    $a['required'] = 'required';                
                if ($cf['field_length'] > 0)
                    $a['maxlength'] = $cf['field_length'];                
                if ($cf['field_regex_validation'])
                    $a['field_regex_validation'] = trim ($cf['field_regex_validation']);
                if ($cf['required'])
                    $a['required'] = 'required';
                $str_attrs = self::attrToString($a);
                $tag = "<x-ui:textbox {$str_attrs} />";                
                break;
        }        
        return $tag;
    }
            
    public function getRequiredFieldInfo($field_type)
    {
        $v_field_length = false;
        $v_field_precision = false;
        $v_field_unsigned = false;
        $invalid_field_type = false;
        $v_integer = false;
        $v_float = false;
        switch ($field_type)
        {
            case 'DATETIME':
            case 'DATE':
            case 'TIME':
            case 'TIMESTAMP':            
            case 'TEXT':
            case 'MEDIUMTEXT':
            case 'LONGTEXT':
                $v_field_unsigned = false;
                $v_field_length = false;                
                $v_field_precision = false;
                break;
            
            case 'VARCHAR':
            case 'CHAR':
                $v_field_unsigned = false;
                $v_field_length = true;                
                $v_field_precision = false;
                break;
            
            case 'FLOAT':
            case 'REAL':
            case 'DOUBLE':
            case 'DECIMAL':
                $v_field_unsigned = false;
                $v_field_length = true;                
                $v_field_precision = true;
                $v_float = true;
                break;
            case 'TINYINT':
            case 'SMALLINT':
            case 'INT':
            case 'MEDIUMINT':
            case 'BIGINT':
                $v_field_unsigned = true;
                $v_field_length = false;                
                $v_field_precision = false;
                $v_integer = true;
                break;                
            default:
                $invalid_field_type = true;
                
        }
        return ['unsigned' => $v_field_unsigned, 'length' => $v_field_length, 'precision' => $v_field_precision, 'integer' => $v_integer, 'float' => $v_float, 'invalid' => $invalid_field_type];
        
    }
    
    public function getFieldTypeString($field_type, $field_length, $field_precision_length, $field_unsigned, $field_default_null = true, $field_default_value = null)
    {
        $result = '';
        $field_length = abs((int) $field_length);
        $field_precision_length = abs((int) $field_length);
        $field_unsigned = is_value_true($field_unsigned);
        $txt_charset = ' CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci ';
        $defv = ' NULL DEFAULT NULL';
        if (!$field_default_null)            
        {
            $sanitized_default = sanitize_string($field_default_value);
            $defv = "NOT NULL DEFAULT '{$sanitized_default}'" ;
        }
        switch ($field_type)
        {
            case 'DATETIME':
            case 'DATE':
            case 'TIME':
            case 'TIMESTAMP':            
            case 'TEXT':
            case 'MEDIUMTEXT':
            case 'LONGTEXT':
                $result = $field_type;
                break;
            
            case 'VARCHAR':
            case 'CHAR':
                $field_length = $field_length > 0 ? $field_length : 255;
                $result = "{$field_type}({$field_length}) {$txt_charset} ";
               
                break;
            
            case 'FLOAT':
            case 'REAL':
            case 'DOUBLE':
            case 'DECIMAL':
                $result = $field_type;
                if ($field_precision_length > 0 && $field_length > 0)
                    $result.= "({$field_length},{$field_precision_length})";
                elseif ($field_length > 0)
                    $result.= "({$field_length})";
                break;
            case 'TINYINT':
            case 'SMALLINT':
            case 'INT':
            case 'MEDIUMINT':
            case 'BIGINT':
                $result = $field_type;
                if ($field_length > 0)
                    $result.= "({$field_length})";
                if ($field_unsigned)
                    $result.= " UNSIGNED ";
                break;                
                
        }
        return $result;
        
    }
    
    
    public function modifyDataValuesBeforeSaveItem($datavalues) {
        
        
        $datavalues = parent::modifyDataValuesBeforeSaveItem($datavalues);
        $rq = $this->getRequiredFieldInfo($datavalues['field_type']);
        if (!$rq['unsigned'])
            $datavalues['field_unsigned'] = null;
        
        
        if (!$rq['length'])
            $datavalues['field_length'] = null;

        if (!$rq['precision'])
            $datavalues['field_precision'] = null;    
        
        $datavalues['field_default'] = trim($datavalues['field_default']);
        if ($datavalues['field_choices_use_sql'])
            $datavalues['field_choices'] = null;
        else
            $datavalues['field_choices_sql'] = null;

        if (!in_array($datavalues['field_box_ui'], $this->field_with_choices))
        {
            $datavalues['field_choices'] = $datavalues['field_choices_sql'] = null;
        }
        
        return $datavalues;
    }
    
    public function ensureCustomFields($table_name, $old_field_name, $field_name, $field_type, $field_length, $field_precision_length, $field_unsigned, $field_default_null, $field_default_value)
    {
        global $SystemDB;
        
        $sql = '';
        $SystemDB->query('TRUNCATE TABLE gk_cache_items');
        if ($SystemDB->tableExists($table_name))
        {
            $is_existing_field = $SystemDB->tableColumnExists($table_name, 'xcf_'.$field_name);
            
            $xt = '';
            $fts = $this->getFieldTypeString($field_type, $field_length, $field_precision_length, $field_unsigned, $field_default_null, $field_default_value);
            if ($old_field_name === '')
            {                 
                
                if ($is_existing_field) // field was added manually
                {
                    $xt = "MODIFY `xcf_{$field_name}` ";
                } else 
                {
                    $xt = "ADD `xcf_{$field_name}`";
                }
                
            } else 
            {
                
                //if ($is_existing_field)
                //{      
                     $is_existing_old_field = $SystemDB->tableColumnExists($table_name, 'xcf_'.$old_field_name);
                     if (!empty($old_field_name) && ($old_field_name != $field_name))
                     {
                         
                         $xt = $is_existing_old_field ? "CHANGE `xcf_{$old_field_name}` `xcf_{$field_name}`" : "ADD `xcf_{$field_name}`";
                     }
                     else
                     {
                         $xt = $is_existing_old_field ? "MODIFY `xcf_{$old_field_name}` " : "ADD `xcf_{$field_name}`";                        
                     }
                     
                //}                 
            }
            $sql = "ALTER TABLE `{$table_name}` {$xt} {$fts} ";
            $SystemDB->query($sql);
            return true;
        } else return false;
    }
    
    public function getValidationErrorListBeforeSaveItem($datavalues) {
        $error_list = parent::getValidationErrorListBeforeSaveItem($datavalues);
        
        if ($datavalues['category_id'] == 0)
            $error_list[] = ___('Category ID cannot be empty');
        else 
        {
            $cat = $this->getCategoryByID($datavalues['category_id']);
            if (empty($cat))
                $error_list[] = ___('Category does not exist');
        
            if (!preg_match('/^[a-zA-Z][a-zA-Z0-9_]{1,60}$/', $datavalues['field_name']))
                $error_list[] = ___('Invalid field name');
            $req_attrs = $this->getRequiredFieldInfo($datavalues['field_type']);
            if ($req_attrs['length'] && ((int) $datavalues['field_length'] <= 0))        
                $error_list[] = sprintf(___('Length must be greater than 0 for %s'), $datavalues['field_type']);
            if ($req_attrs['precision'] && ((int) $datavalues['field_length'] > 30))        
                $error_list[] = sprintf(___('Maximum is 30')); 
            if ($req_attrs['precision'] && ((int) $datavalues['field_precision'] < 0))        
                $error_list[] = sprintf(___('Precision must be greater or equal to 0 for %s'), $datavalues['field_type']);
            if ($req_attrs['invalid'])
                $error_list[] = sprintf(___('Invalid field type specified: %s'), $datavalues['field_type']);
            $defaultval = $datavalues['field_default'];

            if ( ( (int) $datavalues['field_null'] === 0) && !empty($defaultval) && $req_attrs['integer'] && ($defaultval !== '0') && ((int) $defaultval === 0) )
            {                
                $error_list[] = ___('Default value is not an integer');
            }
            if (in_array($datavalues['field_box_ui'], $this->field_with_choices))
            {
                if ($datavalues['field_choices_use_sql'] && empty ($datavalues['field_choices_sql']))
                    $error_list[] = ___('SQL Query must be specified for this UI field type');
                else
                    if (empty ($datavalues['field_choices']))
                        $error_list[] = ___('Choices must be specified for this UI field type');                    
                
            }
            if (empty($error_list))
            {
                $old_field_name = '';
                if ($datavalues['id'] > 0)
                {
                    $old_item = $this->getItemByID($datavalues['id']);
                    $old_field_name = ($old_item != null) ? $old_item['field_name'] : '';
                } else 
                {
                    $existing = $this->getCustomFieldByName($datavalues['field_name']);
                    if ($existing)
                        $error_list[] = ___('Field already exists');
                }
                if (empty($error_list))
                {
                    $precision = array_key_exists('field_precision', $datavalues) ? $datavalues['field_precision']  : '';
                    $this->ensureCustomFields($cat['table_name'], $old_field_name, $datavalues['field_name'], $datavalues['field_type'], $datavalues['field_length'], $precision, $datavalues['field_unsigned'], $datavalues['field_null'], $datavalues['field_default']);
                }
            }
        }
        return $error_list;
        
    }
    /**
     * Run command passed by the main router
     * @param array $command
     * @return bool
     */
    public function Run($command) {
        switch ($command['action']) {
            case 'viewitem':
                $this->viewItemByID($command['id'], $this->cache);
                break;
            case 'viewcategory': $this->viewCategoryByID($command['cid'], $command['pg'], 'date_created', 'DESC', $this->cache);
                break;
            case 'main': $this->viewMainPage($command['pg']);
                break;

            default: return parent::Run($command);
        }

        return RETURN_FULLPAGE;
    }
}
            