<?php
declare(strict_types = 1);

/**
 * Gestion des fichiers en attente de validation.
 *
 * @license https://www.gnu.org/licenses/gpl-3.0.html
 * @link https://www.igalerie.org/
 */
class Pending
{
	/**
	 * Ajoute des fichiers à la galerie
	 * et change leur état (activé, hors ligne).
	 *
	 * @param array $items_id
	 *   Identifiant des fichiers.
	 * @param int $status
	 *   État (0 ou 1).
	 *
	 * @return int
	 *   Retourne le nombre de fichiers affectés
	 *   ou -1 en cas d'erreur.
	 */
	public static function add(array $items_id, int $status): int
	{
		if (!$items_id || !in_array($status, [0, 1]))
		{
			return 0;
		}

		// Récupération des informations utiles de chaque fichier.
		$sql = 'SELECT ip.*,
					   cat_path
				  FROM {items_pending} AS ip
			 LEFT JOIN {categories} USING (cat_id)
				 WHERE pending_id IN (' . DB::inInt($items_id) . ')';
		if (!DB::execute($sql))
		{
			return -1;
		}
		if (!$items = DB::fetchAll('pending_id'))
		{
			return 0;
		}

		// Regroupement des fichiers par albums.
		$albums = [];
		foreach ($items as &$i)
		{
			$albums[$i['cat_path']][] = &$i;
		}

		$albums_notify = [];
		$items_add = 0;
		$pending_path = GALLERY_ROOT . '/pending';
		$pending_delete = [];

		// Lancement d'un scan pour chaque album.
		foreach ($albums as $path => &$files)
		{
			$items_scan = [];

			// On déplace les fichiers du répertoire d'attente
			// vers le répertoire de l'album.
			$album_path = CONF_ALBUMS_PATH . '/' . $path;
			foreach ($files as &$i)
			{
				$filename = File::getSecureFilename($i['pending_file'], $album_path);

				if (file_exists($pending_file = $pending_path . '/' . $i['pending_file'])
				&& File::rename($pending_file, $album_path . '/' . $filename))
				{
					if ($rotate = file_exists($pending_file . '.rotate'))
					{
						File::unlink($pending_file . '.rotate');
					}
					$pending_delete[] = $i['pending_id'];
					$items_scan[$filename] =
					[
						'cat_path' => $i['cat_path'],
						'user_id' => $i['user_id'],
						'item_exif' => $i['pending_exif'],
						'item_iptc' => $i['pending_iptc'],
						'item_xmp' => $i['pending_xmp'],
						'rotate' => $rotate
					];
				}
			}

			if (!$items_scan)
			{
				continue;
			}

			// Initialisation du scan.
			$scan = new Scan();
			if (!$scan->getInit)
			{
				continue;
			}

			// Options de scan.
			$scan->setHttp = TRUE;
			$scan->setHttpFiles = $items_scan;
			$scan->setMailAlert = FALSE;
			$scan->setStatus = $status;

			// Scan du répertoire de l'album courant.
			if ($scan->start($path) === FALSE)
			{
				continue;
			}

			$items_add += $scan->getReport['images_added'] + $scan->getReport['videos_added'];

			if ($items_add)
			{
				$albums_notify[] = $path;
			}
		}

		// On supprime les fichiers de la table {items_pending}.
		$sql = 'DELETE
				  FROM {items_pending}
				 WHERE pending_id IN (' . DB::inInt($pending_delete) . ')';
		if (!DB::execute($sql))
		{
			return -1;
		}

		// On notifie les membres par courriel.
		if ($status)
		{
			$mail = new Mail();
			$mail->notify('items', $albums_notify, Auth::$id,
			[
				'user_id' => Auth::$id,
				'user_name' => Auth::$nickname
			]);
			$mail->send();
		}

		return $items_add;
	}

	/**
	 * Supprime des fichiers.
	 *
	 * @param array $items_id
	 *   Identifiant des fichiers.
	 *
	 * @return int
	 *   Retourne le nombre de fichiers affectés
	 *   ou -1 en cas d'erreur.
	 */
	public static function delete(array $items_id): int
	{
		if (!$items_id)
		{
			return 0;
		}

		$sql = 'FROM {items_pending} WHERE pending_id IN (' . DB::inInt($items_id) . ')';

		// Récupération des noms de fichier.
		if (!DB::execute("SELECT pending_file $sql"))
		{
			return -1;
		}
		if (!$items = DB::fetchCol('pending_file'))
		{
			return 0;
		}

		// On supprime les fichiers en base de données.
		if (!DB::execute("DELETE $sql"))
		{
			return -1;
		}

		// On supprime les fichiers sur le disque.
		foreach ($items as &$file)
		{
			if (file_exists($pending_file = GALLERY_ROOT . '/pending/' . $file))
			{
				File::unlink($pending_file);
			}
		}

		return DB::rowCount();
	}
}
?>