<?php defined('FLATBOARD') or die('Flatboard Community.');
/*
 * Project name: Flatboard
 * Project URL: https://flatboard.org
 * Author: Frédéric Kaplon and contributors
 * All Flatboard code is released under the MIT license.
 * 
 * Classe pour la gestion des plugins dans Flatboard.
 *
 * Fournit des méthodes statiques pour exécuter des hooks de plugins et afficher des notifications
 * de redirection avec messages personnalisés.
 */
class Plugin
{
    /** @var string Chemin du répertoire des données des plugins */
    private const DATA_PLUGIN = DATA_PLUGIN;

    /** @var string Extension des fichiers de données des plugins */
    private const DATA_EXT = '.dat.php';

    /** @var string Classe CSS par défaut pour les notifications */
    private const DEFAULT_ALERT_CLASS = 'alert alert-success';

    /** @var string Délai de redirection automatique en millisecondes */
    private const REDIRECT_DELAY = 1000;
    
    /** @var array Cache statique pour les vérifications de hooks valides */
    private static $hookCache = [];

    /**
     * Constructeur protégé pour empêcher l'instanciation.
     */
    protected function __construct()
    {
        // Classe statique, aucune instanciation nécessaire
    }

    /**
     * Exécute les hooks pour les plugins actifs.
     *
     * @param string $name Nom du hook.
     * @param mixed $param Paramètre optionnel à passer au hook.
     * @param string[] $plugins Liste des plugins actifs (par défaut : global $plugins).
     * @return string Résultat combiné des hooks exécutés.
     * @throws InvalidArgumentException Si le nom du hook est invalide.
     */
    public static function hook(string $name, mixed $param = null, array $plugins = []): string
    {
        global $plugins;
        $plugins = $plugins ?: $plugins;

        if (empty($name) || !preg_match('/^[a-zA-Z0-9_]+$/', $name)) {
            throw new InvalidArgumentException("Nom de hook invalide : '$name'.");
        }

        $out = '';
        foreach ($plugins as $plugin) {
            if (self::isValidHook($name, $plugin)) {
                $out .= self::myHook($name, $plugin, $param);
            }
        }

        return $out;
    }

    /**
     * Vérifie si un hook est valide pour un plugin donné.
     *
     * @param string $hook Nom du hook.
     * @param string $plugin Nom du plugin.
     * @return bool True si le hook est valide, false sinon.
     */
    public static function isValidHook(string $hook, string $plugin): bool
    {
        // Retourne false pour les hooks ou plugins vides/invalides, pour compatibilité avec l'ancienne logique
        if (empty($hook) || empty($plugin) || !preg_match('/^[a-zA-Z0-9_]+$/', $hook . $plugin)) {
            return false;
        }

        $cacheKey = $plugin . '_' . $hook;
        
        // Vérifier le cache
        if (isset(self::$hookCache[$cacheKey])) {
            return self::$hookCache[$cacheKey];
        }
        
        $isValid = function_exists($plugin . '_' . $hook);
        self::$hookCache[$cacheKey] = $isValid;
        
        return $isValid;
    }

    /**
     * Exécute un hook spécifique pour un plugin donné.
     *
     * @param string $hook Nom du hook.
     * @param string $plugin Nom du plugin.
     * @param mixed $param Paramètre optionnel à passer au hook.
     * @return mixed Résultat de l'exécution du hook.
     */
    public static function myHook(string $hook, string $plugin, mixed $param = null): mixed
    {
        $hookFunc = $plugin . '_' . $hook;
        return $hookFunc($param);
    }

    /**
     * Affiche une notification avec redirection optionnelle.
     *
     * @param string $title Titre de la notification.
     * @param string $url URL de redirection.
     * @param string $destination Texte décrivant la destination.
     * @param string $class Classe CSS pour la notification (par défaut : 'alert alert-success').
     * @param bool $autoRedirect Active la redirection automatique (par défaut : true).
     * @param array $lang Tableau des traductions (par défaut : global $lang).
     * @return string HTML de la notification.
     * @throws InvalidArgumentException Si les paramètres sont invalides.
     */
    public static function redirectMsg(
        string $title,
        string $url,
        string $destination,
        string $class = self::DEFAULT_ALERT_CLASS,
        bool $autoRedirect = true,
        array $lang = []
    ): string {
        global $lang;
        $lang = $lang ?: $lang;

        if (empty($title) || empty($url) || empty($destination)) {
            throw new InvalidArgumentException('Le titre, l\'URL et la destination ne peuvent pas être vides.');
        }

        $class = $class ?: self::DEFAULT_ALERT_CLASS;
        $redirectText = $lang['redirect'] ?? 'Rediriger vers';

        $html = sprintf(
            '<div id="alert" class="%s"><div class="text-center"><strong>%s%s</strong><p><a href="%s">%s %s</a></p>',
            htmlspecialchars($class, ENT_QUOTES, 'UTF-8'),
            $autoRedirect ? '<i class="fa fa-spinner fa-pulse fa-lg fa-fw"></i> ' : '',
            htmlspecialchars($title, ENT_QUOTES, 'UTF-8'),
            htmlspecialchars($url, ENT_QUOTES, 'UTF-8'),
            htmlspecialchars($redirectText, ENT_QUOTES, 'UTF-8'),
            htmlspecialchars($destination, ENT_QUOTES, 'UTF-8')
        );

        if ($autoRedirect) {
            $html .= sprintf(
                '<script>setTimeout(function(){window.location.href = "%s"; }, %d);</script>',
                htmlspecialchars($url, ENT_QUOTES, 'UTF-8'),
                self::REDIRECT_DELAY
            );
        }

        $html .= '</div><span class="close small"></span></div>';

        return $html;
    }

    /**
     * Récupère le chemin du fichier de données d'un plugin.
     *
     * @param string $pluginName Nom du plugin.
     * @return string Chemin absolu du fichier de données.
     * @throws InvalidArgumentException Si le nom du plugin est invalide.
     */
    public static function getPluginDBPath(string $pluginName): string
    {
        if (empty($pluginName) || preg_match('/[\/\\\\]/', $pluginName)) {
            throw new InvalidArgumentException("Nom de plugin invalide : '$pluginName'.");
        }

        return self::DATA_PLUGIN . $pluginName . self::DATA_EXT;
    }
}