<?php defined('FLATBOARD') or die('Flatboard Community.');
/*
 * Nom du projet : Flatboard
 * URL du projet : https://flatboard.org
 * Auteur : Frédéric Kaplon et contributeurs
 * Tout le code de Flatboard est publié sous la licence MIT.
 */

class flatDB
{
    /** @var array Cache statique pour les entrées lues */
    private static $readCache = [];
    
    /** @var array Cache statique pour les listes d'entrées */
    private static $listCache = [];
    
    /** @var int Timestamp de la dernière invalidation du cache */
    private static $cacheTime = 0;
    
    /** @var int Durée de vie du cache en secondes (5 secondes) */
    private const CACHE_TTL = 5;

    /**
     * Constructeur protégé car il s'agit d'une classe statique.
     *
     * @access protected
     */
    protected function __construct()
    {
        // Rien ici
    }
    
    /**
     * Invalide le cache si nécessaire.
     */
    private static function invalidateCacheIfNeeded(): void
    {
        $currentTime = time();
        if (($currentTime - self::$cacheTime) > self::CACHE_TTL) {
            self::$readCache = [];
            self::$listCache = [];
            self::$cacheTime = $currentTime;
        }
    }

    /**
     * Obtenir une liste de fichiers dans un répertoire, en ignorant certains fichiers.
     *
     * @param string $dir Chemin du répertoire
     * @return array Liste des noms de fichiers sans extensions
     */
    public static function fdir($dir)
    {
        $ignored = ['.', '..', '.svn', '.git', 'Thumbs.db', 'index.html', '.DS_Store'];
        $files = [];
        
        if ($dh = opendir($dir)) {
            while (false !== ($file = readdir($dh))) {
                if (in_array($file, $ignored)) continue;
                $fileParts = explode('.', $file, 2);
                $files[] = $fileParts[0];
            }
            closedir($dh);
        }
        
        return $files;
    }

    /**
     * Vérifier si un fichier existe dans un répertoire avec une extension spécifique.
     *
     * @param string $file Nom du fichier
     * @param string $dir Chemin du répertoire
     * @param string $ext Extension du fichier
     * @return bool Vrai si le fichier existe, faux sinon
     */
    public static function indir($file, $dir, $ext = '')
    {
        return strpos($file, DS) === false && 
               strpos($file, '.') === false && 
               strpos($file, "\0") === false && 
               file_exists($dir . DS . $file . $ext);
    }

    /**
     * Lire une entrée à partir d'un fichier de données.
     *
     * @param string $type Type d'entrée
     * @param string $file Nom du fichier
     * @return mixed Le contenu du fichier ou null si non trouvé
     */
    public static function readEntry($type, $file)
    {
        self::invalidateCacheIfNeeded();
        
        $cacheKey = $type . ':' . $file;
        
        // Vérifier le cache
        if (isset(self::$readCache[$cacheKey])) {
            return self::$readCache[$cacheKey];
        }
        
        $dataFile = DATA_DIR . $type . DS . $file . '.dat.php';
        if (file_exists($dataFile)) {
            $data = eval('return ' . file_get_contents($dataFile, false, NULL, 14) . ';');
            self::$readCache[$cacheKey] = $data;
            return $data;
        }
        
        return null;
    }

    /**
     * Enregistrer une entrée dans un fichier de données.
     *
     * @param string $type Type d'entrée
     * @param string $file Nom du fichier
     * @param mixed $data Données à enregistrer
     * @return int|false Nombre d'octets écrits ou faux en cas d'échec
     */
    public static function saveEntry($type, $file, $data)
    {
        $result = file_put_contents(
            DATA_DIR . $type . DS . $file . '.dat.php',
            "<?php exit;?>\n" . var_export($data, true),
            LOCK_EX
        );
        
        // Mettre à jour le cache après sauvegarde
        if ($result !== false) {
            $cacheKey = $type . ':' . $file;
            self::$readCache[$cacheKey] = $data;
            // Invalider le cache de liste pour ce type
            unset(self::$listCache[$type]);
        }
        
        return $result;
    }

    /**
     * Supprimer un fichier d'entrée.
     *
     * @param string $type Type d'entrée
     * @param string $file Nom du fichier
     * @return bool Vrai en cas de succès, faux en cas d'échec
     */
    public static function deleteEntry($type, $file)
    {
        $result = unlink(DATA_DIR . $type . DS . $file . '.dat.php');
        
        // Invalider le cache après suppression
        if ($result) {
            $cacheKey = $type . ':' . $file;
            unset(self::$readCache[$cacheKey]);
            // Invalider le cache de liste pour ce type
            unset(self::$listCache[$type]);
        }
        
        return $result;
    }

    /**
     * Lister toutes les entrées d'un type spécifique.
     *
     * @param string $type Type d'entrée
     * @return array Liste des noms de fichiers d'entrée
     */
    public static function listEntry($type)
    {
        self::invalidateCacheIfNeeded();
        
        // Vérifier le cache
        if (isset(self::$listCache[$type])) {
            return self::$listCache[$type];
        }
        
        $list = self::fdir(DATA_DIR . $type);
        self::$listCache[$type] = $list;
        return $list;
    }

    /**
     * Vérifier si un fichier est une entrée valide.
     *
     * @param string $type Type d'entrée
     * @param string $file Nom du fichier
     * @return bool Vrai si valide, faux sinon
     */
    public static function isValidEntry($type, $file)
    {
        return self::indir($file, DATA_DIR . $type, '.dat.php');
    }

    /**
     * Générer un nouveau nom d'entrée unique.
     *
     * @return string Nom d'entrée unique
     */
    public static function newEntry()
    {
        return date('Y-m-dHis') . substr(uniqid(), -5);
    }

    /**
     * Supprimer récursivement tous les fichiers et répertoires dans un répertoire.
     *
     * @param string $dir Chemin du répertoire
     */
    public static function deleteDir($dir)
    {
        if ($handle = opendir($dir)) {
            while (false !== ($file = readdir($handle))) {
                if ($file != '.' && $file != '..') {
                    $filePath = $dir . DS . $file; // Ajout de DS pour le chemin complet
                    if (is_dir($filePath)) {
                        // Si c'est un répertoire, on appelle la méthode récursive
                        self::deleteDir($filePath);
                    } else {
                        // Si c'est un fichier, on le supprime
                        unlink($filePath);
                    }
                }
            }
            closedir($handle);
            // Supprimer le répertoire après avoir supprimé son contenu
            @rmdir($dir);
        }
    }
}