<?php

declare(strict_types=1);

namespace Drupal\RecipeKit\Installer;

use Composer\InstalledVersions;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Extension\ModuleInstallerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Installer\Exception\InstallerException;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Recipe\Recipe;
use Drupal\Core\Recipe\RecipeRunner;
use Drupal\Core\Render\Element\Password;
use Drupal\Core\StringTranslation\Translator\FileTranslation as CoreFileTranslation;
use Drupal\RecipeKit\Installer\Form\SiteSettingsFormDecorator;
use Drupal\RecipeKit\Installer\FormInterface as InstallerFormInterface;

/**
 * Provides hook implementations for profiles built on this kit.
 *
 * @internal
 *   Everything in this class is internal, which means it could be changed in
 *   any way, or removed outright, at any time without warning. It is only meant
 *   to be used by profiles that were generated by this kit. You should not use
 *   it in your own code in any way.
 */
final class Hooks {

  /**
   * The state key under which we track already-applied recipes.
   *
   * @var string
   *
   * @see ::applyRecipes
   * @see ::markRecipeAsApplied
   * @see ::uninstallProfile
   */
  private const string APPLIED_RECIPES_STATE_KEY = 'installer.applied_recipes';

  /**
   * Implements hook_install_tasks().
   */
  public static function installTasks(?array &$install_state = NULL): array {
    // If the container can be altered, wrap the messenger service to suppress
    // certain messages.
    $container = \Drupal::getContainer();
    if ($container instanceof ContainerBuilder) {
      $container->set('messenger', new Messenger(
        \Drupal::messenger(),
      ));

      // Previous versions of Recipe Installer Kit did not pass $install_state
      // to this method, so we need to account for that.
      $install_state ??= $GLOBALS['install_state'];

      // Let the profile declaratively suppress certain messages.
      foreach ($install_state['profile_info']['hide_messages'] ?? [] as $message) {
        if (is_array($message)) {
          assert(count($message) === 2);
          Messenger::rejectPlural(...$message);
        }
        else {
          assert(is_string($message));
          Messenger::reject($message);
        }
      }

      // Also add a translator that will look at translation files for projects
      // other than Drupal core.
      $decorated = $container->get('string_translator.file_translation', ContainerBuilder::NULL_ON_INVALID_REFERENCE);
      if ($decorated instanceof CoreFileTranslation) {
        \Drupal::translation()->addTranslator(
          FileTranslation::createFromInstance($decorated),
        );
      }
    }

    // If one of the configuration forms chose a finish URL, use it at the end
    // of the install process. For backwards compatibility, the finish URL
    // specified by the profile itself (if any) takes precedence. We can't use
    // $install_state directly here because it might not be a reference.
    // @see install_drupal()
    // @see \Drupal\RecipeKit\Installer\Form\SiteTemplateForm::submitForm()
    $GLOBALS['install_state']['profile_info']['distribution']['install']['finish_url'] ??= $install_state['parameters']['finish_url'] ?? NULL;

    return [
      'uninstall_profile' => [
        // As a final task, uninstall the install profile.
        'function' => static::class . '::uninstallProfile',
      ],
    ];
  }

  /**
   * Implements hook_install_tasks_alter().
   */
  public static function installTasksAlter(array &$tasks, array $install_state): void {
    $insert_before = function (string $key, array $additions) use (&$tasks): void {
      $key = array_search($key, array_keys($tasks), TRUE);
      if ($key === FALSE) {
        return;
      }
      // This isn't very clean, but it's the only way to positionally splice
      // into an associative (and therefore by definition unordered) array.
      $tasks_before = array_slice($tasks, 0, $key, TRUE);
      $tasks_after = array_slice($tasks, $key, NULL, TRUE);
      $tasks = $tasks_before + $additions + $tasks_after;
    };

    // Set up any pre-database information collection forms.
    $forms = [];
    foreach ($install_state['profile_info']['forms'] ?? [] as $class) {
      assert(is_a($class, InstallerFormInterface::class, TRUE));
      $forms[$class] = $class::toInstallTask($install_state);
    }
    $insert_before('install_settings_form', $forms);

    // Wrap the database settings form, because we cannot rely on the installer
    // to invoke the profile's hook_form_alter().
    $tasks['install_settings_form']['function'] = SiteSettingsFormDecorator::class;

    $configure_form_task = $tasks['install_configure_form'];
    unset(
      $tasks['install_install_profile'],
      $tasks['install_configure_form'],
    );
    $insert_before('install_profile_modules', [
      'install_install_profile' => [
        'function' => Hooks::class . '::installProfile',
      ],
      'install_configure_form' => $configure_form_task,
    ]);

    // Set English as the default language; we support changing it mid-stream.
    // We can't use the passed-in $install_state here because it's not passed
    // by reference.
    $GLOBALS['install_state']['parameters'] += ['langcode' => 'en'];

    // If translations will be downloaded, ensure that we also download the
    // translations for the install profile and this kit.
    $tasks['install_download_translation']['function'] = static::class . '::downloadTranslations';

    // Wrap the install_profile_modules() function, which returns a batch job, and
    // add all the necessary operations to apply the chosen template recipe.
    $tasks['install_profile_modules']['function'] = static::class . '::applyRecipes';
  }

  /**
   * Downloads translations for the install profile and this kit.
   *
   * This wraps `install_download_translation()` and downloads core translations
   * last. If the core translations fail to download, the install process will
   * stop with an exception.
   *
   *  @return mixed
   *  Return value from `install_download_translation()`.
   */
  public static function downloadTranslations(array &$install_state): mixed {
    // Temporarily disable the interactive installer so that
    // `install_download_translation()` won't reload the page.
    $was_interactive = $install_state['interactive'];
    $install_state['interactive'] = FALSE;

    // Always download the translation for this kit.
    $projects = ['recipe_installer_kit'];

    // If the install profile is part of a specific project, download its
    // translation, too.
    if (isset($install_state['profile_info']['project'])) {
      $projects[] = $install_state['profile_info']['project'];
    }

    $original_server_pattern = $install_state['server_pattern'];
    foreach ($projects as $name) {
      try {
        // Construct a download URL for the package. We can't rely on
        // `install_download_translation()` to do this for us, because it is
        // hard-coded to download the translation for core.
        $install_state['server_pattern'] = strtr($original_server_pattern, [
          '%project' => $name,
          // @todo Make this work with packages that aren't in the `drupal`
          //   vendor namespace.
          '%version' => InstalledVersions::getPrettyVersion('drupal/' . $name),
        ]);
        install_download_translation($install_state);
      }
      catch (\OutOfBoundsException) {
        // Apparently the project is not installed by Composer, so we cannot
        // reliably figure out its version to download a translation.
      }
      catch (InstallerException) {
        // If there's an error, `install_display_requirements()`, which is
        // called by `install_download_translation()`, will throw. That's a
        // pity, but it's probably better to just keep chugging along with
        // missing translations.
      }
      finally {
        $install_state['server_pattern'] = $original_server_pattern;
      }
    }

    // Download core translations as normal.
    $install_state['interactive'] = $was_interactive;
    return install_download_translation($install_state);
  }

  /**
   * Decorates install_install_profile(), ensuring User is installed first.
   */
  public static function installProfile(): void {
    // We'll need User to configure the site and administrator account.
    \Drupal::service(ModuleInstallerInterface::class)->install(['user']);

    // Officially install the profile so that its behaviors and visual overrides
    // will be in effect for the remainder of the install process. This also
    // ensures that the administrator role is created and assigned to user 1 in
    // the next step.
    global $install_state;
    install_install_profile($install_state);
  }

  /**
   * Implements hook_form_alter().
   */
  public static function formAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    if ($form_id === 'install_configure_form') {
      self::installConfigureFormAlter($form);
    }
  }

  /**
   * Implements hook_form_alter() for install_configure_form.
   *
   * @see \Drupal\Core\Installer\Form\SiteConfigureForm
   */
  private static function installConfigureFormAlter(array &$form): void {
    global $install_state;

    $form['site_information']['#type'] = 'container';

    // If we collected the site name in a previous step, don't show it now.
    if (array_key_exists('site_name', $install_state['parameters'])) {
      $form['site_information']['site_name'] = [
        '#type' => 'hidden',
        '#default_value' => $install_state['parameters']['site_name'],
      ];
    }

    // Use a custom value callback to set the site email. Normally this is a
    // required field, but setting `#access` to FALSE seems to bypass that.
    $form['site_information']['site_mail']['#access'] = FALSE;
    $form['site_information']['site_mail']['#value_callback'] = static::class . '::setSiteMail';

    $form['admin_account']['#type'] = 'container';
    // `admin` is a sensible name for user 1.
    $form['admin_account']['account']['name'] = [
      '#type' => 'hidden',
      '#default_value' => 'admin',
    ];
    $form['admin_account']['account']['mail'] = [
      '#type' => 'email',
      '#title' => t('Email'),
      '#required' => TRUE,
      '#default_value' => $install_state['forms']['install_configure_form']['account']['mail'] ?? '',
      '#weight' => 10,
    ];
    $form['admin_account']['account']['pass'] = [
      '#type' => 'password',
      '#title' => t('Password'),
      '#required' => TRUE,
      '#default_value' => $install_state['forms']['install_configure_form']['account']['pass']['pass1'] ?? '',
      '#weight' => 20,
      '#value_callback' => static::class . '::passwordValue',
    ];

    // Hide the timezone selection. Core automatically uses client-side
    // JavaScript to detect it, but we don't need to expose that to the user.
    // But the JavaScript expects the form elements to look a certain way, so
    // hiding the fields visually is the correct approach here.
    // @see core/misc/timezone.js
    $form['regional_settings']['#attributes']['class'][] = 'visually-hidden';
    // Don't allow the timezone selection to be tab-focused.
    $form['regional_settings']['date_default_timezone']['#attributes']['tabindex'] = -1;
  }

  public static function passwordValue(&$element, $input, FormStateInterface $form_state): mixed {
    // Work around the fact that Drush and `drupal install`, which submit this
    // form programmatically, assume the password is a password_confirm element.
    if (is_array($input) && $form_state->isProgrammed()) {
      $input = $input['pass1'];
    }
    return Password::valueCallback($element, $input, $form_state);
  }

  /**
   * Custom submit handler to update the site email.
   */
  public static function setSiteMail(array &$element, $input, FormStateInterface $form_state): string {
    // We can't use $form_state->getValues() because we're a value callback,
    // and therefore still in the middle of populating $form_state's values!
    $user_input = $form_state->getUserInput();
    return $user_input['account']['mail'] ?? strval($input);
  }

  /**
   * Runs a batch job that applies the chosen set of recipes.
   *
   * @param array $install_state
   *   An array of information about the current installation state.
   *
   * @return array
   *   The batch job definition.
   */
  public static function applyRecipes(array &$install_state): array {
    // Apply required recipes first, followed by any additional recipes the user
    // has chosen (i.e., via our forms).
    $recipes_to_apply = array_merge(
      $install_state['profile_info']['recipes']['required'] ?? [],
      $install_state['parameters']['recipes'] ?? [],
    );

    // If the installer ran before but failed mid-stream, don't reapply any
    // recipes that were successfully applied.
    $recipes_to_apply = array_diff(
      $recipes_to_apply,
      \Drupal::state()->get(static::APPLIED_RECIPES_STATE_KEY, []),
    );

    $recipes_to_apply = array_unique($recipes_to_apply);
    // If we've already applied all the chosen recipes, there's nothing to do.
    // Since we only start applying recipes once `install_profile_modules()` has
    // finished, we can be safely certain that we already did that step.
    if (empty($recipes_to_apply)) {
      return [];
    }

    $batch = install_profile_modules($install_state);

    $recipe_operations = [];
    foreach ($recipes_to_apply as $name) {
      $recipe = static::getRecipePath($name);
      $recipe = Recipe::createFromDirectory($recipe);
      $recipe_operations = array_merge($recipe_operations, RecipeRunner::toBatchOperations($recipe));
      $recipe_operations[] = [[static::class, 'markRecipeAsApplied'], [$name]];
    }

    // Only do each recipe's batch operations once.
    foreach ($recipe_operations as $operation) {
      if (!in_array($operation, $batch['operations'], TRUE)) {
        $batch['operations'][] = $operation;
      }
    }
    return $batch;
  }

  /**
   * Marks a particular recipe as having been applied.
   *
   *  This is done to increase fault tolerance. On hosting plans that don't have
   *  a ton of RAM or computing power to spare, the possibility of the installer
   *  timing out or failing in mid-stream is increased, especially with a big,
   *  complex distribution like Drupal CMS. Tracking the recipes which have been
   *  applied allows the installer to recover and "pick up where it left off",
   *  without applying recipes that have already been applied successfully. Once
   *  the install is done, the list of recipes is deleted.
   *
   * @param string $name
   *   The recipe's Composer package name.
   *
   * @see ::uninstallProfile()
   */
  public static function markRecipeAsApplied(string $name): void {
    $key = static::APPLIED_RECIPES_STATE_KEY;
    $list = \Drupal::state()->get($key, []);
    $list[] = $name;
    \Drupal::state()->set($key, array_unique($list));
  }

  /**
   * Uninstalls the install profile, as a final step.
   *
   * @see drupal_install_system()
   */
  public static function uninstallProfile(): void {
    global $install_state;
    \Drupal::service(ModuleInstallerInterface::class)->uninstall([
      $install_state['parameters']['profile'],
    ]);

    // The install is done, so we don't need the list of applied recipes anymore.
    \Drupal::state()->delete(static::APPLIED_RECIPES_STATE_KEY);

    // Clear all previous status messages to avoid clutter.
    \Drupal::messenger()->deleteByType(MessengerInterface::TYPE_STATUS);

    // Invalidate the container in case any stray requests were made during the
    // install process, which would have bootstrapped Drupal and cached the
    // install-time container, which is now stale (during the installer, the
    // container cannot be dumped, which would normally happen during the
    // container rebuild triggered by uninstalling this profile). We do not want
    // to redirect into Drupal with a stale container.
    \Drupal::service('kernel')->invalidateContainer();
  }

  /**
   * @internal
   *   This method is internal, which means it could be changed in any way, or
   *   removed at any time, without warning. Don't rely on it.
   */
  public static function getRecipePath(?string $name = NULL): string {
    try {
      return InstalledVersions::getInstallPath($name);
    }
    catch (\OutOfBoundsException $e) {
      // Composer doesn't know where it is, so try to extrapolate the path by
      // reading `composer.json`.
      ['install_path' => $project_root] = InstalledVersions::getRootPackage();
      $file = $project_root . DIRECTORY_SEPARATOR . 'composer.json';
      $data = file_get_contents($file);
      $data = json_decode($data, TRUE, flags: JSON_THROW_ON_ERROR);

      $installer_paths = $data['extra']['installer-paths'] ?? [];
      foreach ($installer_paths as $path => $criteria) {
        // The first configured install path which matches the criteria is the
        // one we'll use, since that is what Composer would also do.
        if (in_array($name, $criteria, TRUE) || in_array('type:' . Recipe::COMPOSER_PROJECT_TYPE, $criteria, TRUE)) {
          $path = $project_root . DIRECTORY_SEPARATOR . $path;

          return $name
            ? str_replace(['{$vendor}', '{$name}'], explode('/', $name, 2), $path)
            : str_replace('{$name}', '', $path);
        }
      }
      // We couldn't figure it out; throw the original exception.
      throw $e;
    }
  }

}
